#!/usr/local/bin/perl
# getcatalog.pl (original: get2mass.pl)
#   -- To retrieve 2MASS/WISE all sky survey catalog data
#      from the VizieR upon request
# Usage: getwise.pl 05:25:33.4 -69:19:30.0 10.0
# Created: Yoshifusa Ita, 19-Dec-05
# Modified by Wada, 2007/07/28 look for backup sites, options, help...
#                              jis->euc
# Modified by Wada, 2007/08/05 connect-timeout
# Modified by Wada, 2007/10/10 maximum time for the job
# Modified by Nakamura, 2013/02/19 adopted to WISE catalog
#             F. Egusa, 2014/03/17 according to F. Usi suggestion

use LWP::UserAgent;
use Getopt::Std;

# default verbose level:  =zero no debug message, >zero....
$debug=0;
#$debug=1;

#$time_out=0; #try forever
$time_out=30; # wait 30 sec to connect, then try the other site..

$time_max=3600; # wait 3600 sec to finish the job, then try the other site..

# default WISE server site 
#$site="vizier.u-strasbg.fr";
$site="vizier.nao.ac.jp";
	  
# backup WISE server sites 
@sites=qw(vizier.u-strasbg.fr
	  vizier.nao.ac.jp 
	  vizier.hia.nrc.ca 
	  vizier.ast.cam.ac.uk 
	  vizier.iucaa.ernet.in 
	  data.bao.ac.cn 
	  vizier.cfa.harvard.edu 
	  vizier.ukirt.jach.hawaii.edu);

# default output file name
$out_file="2mass.radec";

sub help{
    printf(STDERR "Usage:   catalog.pl catalog_name alpha(deg) delta(deg) fov(arcmin)\n");
    printf(STDERR "example: getcatalog.pl 2MASS 12.34 -12.34 10.0\n");
    printf(STDERR "Options:\n");
    printf(STDERR "  -h: print help\n");
    printf(STDERR "  -s server: catalog site name, the default is %s\n",$site);
    printf(STDERR "  -o output: output catalog file name, the default is %s\n",$out_file);
    printf(STDERR "  -v verbose_level: output debug message if not 1, the default is %d\n",$debug);
    printf(STDERR "  -c connection_timeout: the default is %d\n",$time_out);
}

sub formatcatalog{
    my @lines = split(/\n/, shift);
    open(OUT, "> 2mass.radec") || die "Cannot write 2mass.radec";

    foreach $line (@lines){
        if($line =~ /^\d/){
            @buf1=split(/\t/,$line);
            printf OUT ("%04d %f %f %6.3f %6.3f\n",
                $buf1[0], # id
                $buf1[1], # ra [degree]
                $buf1[2], # dec [degree]
                $buf1[3], # mag [mag]
                $buf1[4], # err [mag]
            );
#            print split(/\s/, $line);
        }
    }
}

getopts('hs:o:v:c:m:',\%options); #%options(I$,L5$$>$B!&n&(IO!"(Bopt_d(I$KCM$,@_D$Bt&(I5$$B!&!&(B
if(defined $options{h}){
    &help();
    exit(0);
}
if(defined $options{s}){
    $site=$options{s};
}
if(defined $options{o}){
    $out_file=$options{o};
}
if(defined $options{c}){
    $time_out=$options{c};
}
if(defined $options{v}){
    $debug=$options{v};
}

if(@ARGV < 4){
    die "Usage: getcatalog.pl catalog(2MASS/WISE) 12.34 -12.34 10.0\n";
}

$catalog = $ARGV[0];
chomp $catalog;
$ra  = $ARGV[1]; # ra  [degree]
chomp $ra;
$dec = $ARGV[2]; # dec [degree]
chomp $dec;
$box = $ARGV[3]; # box [arcmin]
chomp $box;
printf(STDERR "%s %s %s %s\n",$catalog, $ra,$dec,$box) if($debug);

unshift @sites, $site;

if($catalog =~ /2MASS/i){
    $ccode = "II/246";
    $outrow = "RAJ2000,DEJ2000,Kmag,e_Kmag";
    $sortkey = "Kmag";
    $addcond = ""
}elsif($catalog =~ /WISE/i){
## from F. Usui (2014/03/17)
#    $ccode = "II/311";
    $ccode = "II/328";
    $outrow = "RAJ2000,DEJ2000,W3mag,e_W3mag";
    $sortkey = "W3mag";
    $addcond = "&W3mag=<12.5"
}else{
    die "Catalog name is invalid (choose 2MASS or WISE)."
}

for($i=0;$i<$#sites;$i++){
    print "Now retrieving and downloading catalog data\n";
## from F. Usui (2014/03/17)
#    my $url = sprintf("http://%s/viz-bin/asu-acl?-source=%s&-c.ra=%s&-c.dec=%s&-c.bm=%s&-out.max=%s&-out=%s&-sort=%s%s",
#                      $sites[i], $ccode, $ra, $dec, $box, 99999, $outrow, $sortkey, $addcond);
    my $url = sprintf("http://%s/viz-bin/asu-acl?-source=%s&-c.ra=%s&-c.dec=%s&-c.bm=%s&-out.max=%s&-out=%s&-sort=%s%s",
                      $sites[$i], $ccode, $ra, $dec, $box, 99999, $outrow, $sortkey, $addcond);
#    $result=system("curl --connect-timeout $time_out  --max-time $time_max -# -o $out_file -O $url");
#    $result/=256;
#    printf(STDERR "exit code is %d\n",$result) if($debug);
    my $ua = LWP::UserAgent->new;
    $ua->timeout(30);
    my $response = $ua->get($url);
    if($response->is_success){
        formatcatalog($response->content);
        exit;
    }
    if($i==0){ printf(STDERR "Failed to access WISE server. Try backups\n"); }
}
printf(STDERR "Failed to access backup catalog servers.\n");
exit;

