## mk_el.cl -- to create a pattern of the Earth shine light (EL)
## All the images in the input list must be 
##  1) consistent in the band & exposure
##  2) not crossing the critical date (2007/01/08)
##  3) reduced through the 'prepipeline' & 'pipeline1' but before flat fielding
##  4) not containing extended objects

## needs update for new MIR-S flat

## created: 2010/08/19, Fumi Egusa
## modified: 2010/10/19 (F. Egusa), assuming the use after 'pipeline1'
##                                  using old flat for sky
##           2010/10/20 (F. Egusa), to be consistent with 'sub_el'

## products of this script
##  filelist".EL.fits": template for the Earth light (sky subtracted and mask applied)
##  filelist".sky.fits": template for the sky pattern (based on the old flat)
##  filelist".mask.fits": mask file for the input list
##  "k"filename: sky subtracted and mask applied images
##  "rk"filename: ratio of observed/EL
##  "r"filename: ratio of observed/sky

procedure mk_el(filelist)

string filelist {prompt="a list of files to be analyzed"}
bool deltmp=no {prompt="delete temporary files?"}
int niter {0, prompt="number of iteration: 0=no iteration"}
real blankval {-9999.9,max=-1000., prompt="blank value"}
real ratiolimit {0.05, prompt="lower limit to scaling factor for EL pattern"}

struct *list0

begin

 string flist
 int nitr
 real blank
 string fname,lname,band,band1,detec
 string libpath,flat,flatpath
 string fEL,fmin,fmax,fsky,fmask,fapix,fslit,region
 int n,nn,date,mindate,maxdate,expid,expid1
 real ratio,thratio,vsky

## get query parameters
 flist=filelist
 nitr=niter
 blank=blankval
 thratio=ratiolimit

## default parameters
 if ((strlstr(".list",flist) == strlen(flist)-4) || (strlstr(".vsky",flist) == strlen(flist)-4)) {
    lname=substr(flist,1,strlen(flist)-5)
 } else {
    lname=flist
 }
 fmask=lname//".mask.fits"
 fsky=lname//".sky.fits"
 fEL=lname//".EL.fits"


## check the input list
 n=0
 list0=flist
 mindate=30000000 #initial guess
 maxdate=0 #initial guess
 while (fscan(list0,fname) != EOF) {
    n=n+1
    if (access("tmp.dat")) del("tmp.dat")
    hselect(fname,"DATE-REF",yes, > "tmp.dat")
    print("! cut -b 1-10 tmp.dat | sed s/-/''/g") | cl | scan(date)
    mindate=min(mindate,date)
    maxdate=max(maxdate,date)
    hselect(fname,"FILTER",yes) | scan(band)
    hselect(fname,"EXPID",yes) | scan(expid)
    if (expid > 1) expid=2
    if (n == 1) {
       band1=band
       expid1=expid
       hselect(fname,"DETECTOR",yes) | scan(detec)
    } else {
       if ((mindate <= 20070108) && (maxdate > 20070108)) {
      	  print("### ERROR: observing dates crossing over the critical date (2007/01/08)!")
      	  bye
       }
       if (band != band1) {
      	  print("### ERROR: input files are not consistent in observing band!")
      	  bye
       }
       if (expid != expid1) {
       	  print("### ERROR: input files are not consistent in exposure time!")
      	  bye
       }
    }
 }

## read the input list
 if (access("filename.list")) del("filename.list")
 print("! grep -v '#' "//flist//" | awk '{print $1}' > filename.list") | cl

## define masks, flat, & region for linmatch
 libpath=osfn("irclib$")
 if (date > 20070108) {
    flatpath="flat/soramame-nashi/"
 } else {
    flatpath="flat/soramame-ari/"
 }
 if (detec == "NIR") {
    fapix=libpath//"anomalous_pix/outlier_NIR.fits"
    fslit=libpath//"slit_mask/slit_mask_nir.fits"
    region="[1:412,1:384]"
    if (band == "N2") flat=libpath//flatpath//"N2.fits"
    if (band == "N3") flat=libpath//flatpath//"N3.fits"
    if (band == "N4") flat=libpath//flatpath//"N4.fits"
 } else if (detec == "MIRS") {
    fapix=libpath//"anomalous_pix/outlier_MIRS.fits"
    fslit=libpath//"slit_mask/slit_mask_mirs.fits"
    region="[27:256,1:256]"
    if (band == "S7") flat=libpath//flatpath//"S7.fits"
    if (band == "S9W") flat=libpath//flatpath//"S9W.fits"
    if (band == "S11") flat=libpath//flatpath//"S11.fits"
 } else if (detec == "MIRL") {
    fapix=libpath//"anomalous_pix/outlier_MIRL.fits"
    fslit=libpath//"slit_mask/slit_mask_mirl.fits"
    region="[1:236,1:256]"
    if (band == "L15") flat=libpath//flatpath//"L15.fits"
    if (band == "L18W") flat=libpath//flatpath//"L18W.fits"
    if (band == "L24") flat=libpath//flatpath//"L24.fits"
 }
 if (access(fmask)) imdel(fmask)
 imexpr("a*(b+c/d)",fmask,fapix,1,fslit,9999.9,"","","","","","","","","","","","","","","","","","","","","","",outtype="int",ver-)


 n = 0

## sky fitting
skyfit:
 if (n == 0) { #find initial templates for sky & EL
    if (access("tmp.dat")) del("tmp.dat")
    if (access("fsky.list")) del("fsky.list")
    imstat("@filename.list",lower=blank+0.1,nclip=5,format-,fields="im,mean,mode,npix", > "tmp.dat")
    print("! sort -k 3 -n tmp.dat | awk '{print $1}' | head -1") | cl | scan(fmin)
    print("! sort -k 3 -n tmp.dat | awk '{print $1}' | tail -1") | cl | scan(fmax)
    print(fmin, >> "fsky.list")
 } else {
    fmin="tmpsky.fits"
    if (access(fmin)) imdel(fmin)
    combine("@fsky.list",fmin,combine="median",reject="sigclip",blank=blank,lthreshold=blank+1.)
 }
 print("### template for sky: "//fmin)

## sky fit with mode of ratio
 if (access(fsky)) imdel(fsky)
 if (access("tmp1.fits")) imdel("tmp1.fits")
 imexpr("(c>0) ? a/b/c : d","tmp1.fits",fmin,flat,fmask,blank,"","","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
 imstat("tmp1.fits",lower=blank+0.1,fields='mode',nclip=5,format-) | scan(ratio)
 print("###  mode of ratio (sky template/flat): "//ratio)
 imexpr("(c>0) ? a*b/c : d",fsky,flat,ratio,fmask,blank,"","","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
 imstat(fsky,fields="mean",lower=blank+0.1,nclip=5,format-) | scan(vsky)
 print("###  constant sky value: "//vsky)

 if (access("skyratio.list")) del("skyratio.list")
 if (access("skyratio.imst")) del("skyratio.imst")
 list0="fsky.list"
 while (fscan(list0,fname) != EOF) {
    if (access("r"//fname)) imdel("r"//fname)
    imexpr("(c>0) ? a/b/c : d","r"//fname,fname,fsky,fmask,blank,"","","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
    print("r"//fname, >> "skyratio.list")
 }
 imstat("@skyratio.list",fields='im,npix,mode,mean,stddev,min,max',nclip=5,lower=blank+0.1, > "skyratio.imst")


## sky subtraction
 if (access("subsky.list")) del("subsky.list")
 print("! cat filename.list | awk '{print \"k\"$1}' > subsky.list") | cl
 list0="filename.list"
 while (fscan(list0,fname) != EOF) {
    if (access("k"//fname)) imdel("k"//fname)
    imexpr("(c>0) ? (a-b)/c : d","k"//fname,fname,fsky,fmask,blank,"","","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
 }


## make EL
 if (access(fEL)) imdel(fEL)
 if (access("tmp.list")) del("tmp.list")
 nn=0
 if (n == 0) {
    imcopy("k"//fmax,fEL,ver-)
    print("### template for EL: k"//fmax)
 } else {
    list0="fEL.list"
    while (fscan(list0,fname,ratio) != EOF) {
       if (ratio > 0.) {
       	  nn=nn+1
	  if (access("tmp"//nn//".fits")) imdel("tmp"//nn//".fits")
       	  imexpr("(c>0) ? a/b/c : d","tmp"//nn//".fits",fname,ratio,fmask,blank,"","","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
	  print("tmp"//nn//".fits", >> "tmp.list")
       }
    }
    combine("@tmp.list",fEL,combine="median",reject="sigclip",blank=blank,lthreshold=blank+1.)
    print("### template for EL: "//fEL)
 }


## EL fitting & calculate the ratio
 if (access("ELratio.list")) del("ELratio.list")
 print("! cat subsky.list | awk '{print \"r\"$1}' > ELratio.list") | cl

 if (access("fsky.list")) del("fsky.list")
 if (access("fEL.list")) del("fEL.list")
 list0="subsky.list"
 while (fscan(list0,fname) != EOF) {
    if (access("r"//fname)) imdel("r"//fname)
    imexpr("(d>0) ? a/c/d : e","r"//fname,fname,"",fEL,fmask,blank,"","","","","","","","","","","","","","","","","","","","","",outtype="real",ver-)
    imstat("r"//fname,fields="mode",lower=blank+0.1,nclip=5,format-) | scan(ratio)
    if (ratio < thratio) ratio=0.
    if (ratio == INDEF) ratio=0.
    if ((n==0) && (fname == "k"//fmax)) ratio=1.
    print("###  EL scaling factor for "//fname//": "//ratio)
    if (ratio == 0.) { 
       print(substr(fname,2,strlen(fname)), >> "fsky.list")
    } else {
       print(fname//"  "//ratio, >> "fEL.list")
    }
 }

 if (access("ELratio.imst")) del("ELratio.imst")
 imstat("@ELratio.list",fields='im,npix,mode,mean,stddev,min,max',nclip=5,lower=blank+0.1, > "ELratio.imst")


## for iteration
 n=n+1
 if (nitr == 0) goto cleanfiles
 if (n > nitr) goto cleanfiles

 print("")
 print("### "//n//"th iteration ###")
 print("")

 goto skyfit


## deleting temporary files
cleanfiles:
 if (deltmp) {
    imdel("@subsky.list")
    del("subsky.list")
 }

## print summary
 print("")
 print("### For checking results")
 print("###  final fits for the sky pattern: "//fsky)
 print("###  final fits for the EL pattern: "//fEL)
 print("###  list of FITS for the ratio of sky template/flat: skyratio.list")
 print("###  statistics for the above FITS: skyratio.imst")
 print("###  list of FITS for the ratio of observed/template EL: ELratio.list")
 print("###  statistics for the above FITS: ELratio.imst")
 print("")

## end of the script
end
