// convert2mass2.c
// To convert 2mass alpha delta to wcs xy.
// Created: Yoshifusa Ita, 19-Dec-05
// Updated: Tomohiko Nakamura, 28-May-13
// gcc -Wall -O2 -o convert2mass2 convert2mass2.c -lm
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#define PI M_PI
#define deg2rad ( PI/180. )
#define rad2deg ( 180./PI )
#define NDIM 3
#define FRDMIN  ( 180.0*60.0/PI ) 

void wcsxy( double alpha_c, double delta_c, double alpha, double delta, double *wcs_x, double *wcs_y );
void eq2ec(double alpha, double delta, double *lambda, double *beta);
char *stringcp( char *s1, char *s2, int l );
void rotmat( int iaxis, double theta, double *r );
void txexyz( double xi, double eta, double *x );
void mult0( double *r, double *x, double *y );

int main(int argc, char **argv){
  FILE *finp, *fout;
  double alpha, delta;
  double alpha0, delta0;
  double lambda, beta;
  double lambda0, beta0;
  double x0, y0, x, y;
  char text[500];
  char id[50];
  float kmag, kerr;
  float pfov; // FoV per pixel

  if(argc < 6){
    puts("Usage: a.out input.file output.file crval1(ra) crval2(dec) crpix1 crpix2 pfov[arcsec/pix]");
    puts("EX.  : a.out input.file output.file 123.456 -123.456 120 128 2.34");
    exit(1);
  }

  // coordinates at the center of the image
  alpha0 = atof(argv[3]); delta0 = atof(argv[4]);
  x0 = atof(argv[5]); y0 = atof(argv[6]);
  pfov = atof(argv[7]);

  alpha0 *= deg2rad;
  delta0 *= deg2rad;
  eq2ec(alpha0, delta0, &lambda0, &beta0);

  finp = fopen(argv[1], "r");  
  fout = fopen(argv[2], "w");
  while( fgets(text,sizeof(text),finp) != NULL ){
    sscanf(text, "%s %lf %lf %f %f", id, &alpha, &delta, &kmag, &kerr);
    alpha *= deg2rad;
    delta *= deg2rad;
    eq2ec(alpha, delta, &lambda, &beta);
    wcsxy(alpha0, delta0, alpha, delta, &x, &y);
    fprintf(fout, "%s %11.6lf %11.6lf %10.4lf %10.4lf %7.3f %7.3f\n", id, alpha*rad2deg, delta*rad2deg, -x/pfov+x0, y/pfov+y0, kmag, kerr);
  }
  fclose(finp);
  fclose(fout);

  return 0;
}

void wcsxy( double alpha_c, double delta_c, double alpha, double delta, double *wcs_x, double *wcs_y ){ 
/* ---------------------------------------------------------------------
 Calculate x and y from alpha and deta 
	alpha_c	: ceter of Right Ascention (radian) 
	delta_c	: ceter of Decrination  (radian) 
	alpha	: Right Ascention (radian) of object 
	delta	: Decrination (radian) of object 
	wcs_x	: relative x value from the center on the plotted image 
		  East (left) is + and West (right) is -
	wcs_y	: relative y value from the center on the plotted image 
		  North (top) is + and South (bottom) is -
 written by T.Tanabe
*----------------------------------------------------------------------*/ 
  double theta, dx, dy ; 
  double phi_p, R, phi = 0.0 ; 
  double theta1 ; 
  double x[3], y[3], z[3], r[9] ;
  double  eps = 1.0e-16 ;

  if ( fabs( delta_c - PI/2.0 ) < eps ) { 
    phi_p = 0.0 ; 
  } 
  else { 
    phi_p = PI ; 
  } 
  
  txexyz( alpha, delta, x ) ; 
  rotmat( 3, alpha_c + PI/2.0, r ) ; 
  mult0( r, x, y ) ;
  rotmat( 1, PI/2.0 - delta_c, r ) ; 
  mult0( r, y, z ) ;
  rotmat( 3, PI/2.0 - phi_p, r ) ; 
  mult0( r, z, x ) ;
  
  if ( fabs( x[2] - 1.0 ) < eps ) { 
    theta = PI / 2.0 ; 
  } 
  else if ( fabs( x[2] + 1.0 ) < eps ) { 
    theta = -PI / 2.0 ; 
  } 
  else { 
    theta = asin( x[2] ) ; 
  } 
  theta1 = PI / 2.0 - theta ; 
  R = FRDMIN / tan( theta ) ; 
  if ( fabs( x[0] ) < eps ) { 
    if ( fabs( x[1] ) < eps ) { 
      phi = 0.0 ; 
    } 
    else if ( x[1] > 0.0 ) { 
      phi = PI/2.0 ; 
    } 
    else if ( x[1] < 0.0 ) { 
      phi = -PI/2.0 ; 
    } 
  } 
  else if ( x[0] > 0.0 ) { 
    phi = atan( x[1] / x[0] ) ; 
  } 
  else if ( x[0] < 0.0 ) { 
    phi = atan( x[1] / x[0] ) + PI ; 
  } 
  if ( phi < 0.0 ) { 
    phi = phi + 2.0 * PI ; 
  } 
  
  dx = R * sin( phi ) ; 
  dy = -R * cos( phi ) ; 
  
  if ( theta1 > PI/2.0 ) { 
    dx = 5400.0 ; 
    dy = 5400.0 ; 
  } 
  
  *wcs_x = dx*60. ; 
  *wcs_y = dy*60. ; 
}

void rotmat( int iaxis, double theta, double *r ) { 
  int i, j ; 
  double c, s ; 

  for ( i = 0 ; i < NDIM ; i++ ) { 
    for ( j = 0 ; j < NDIM ; j++ ) 
      r[i*NDIM+j] = 0.0 ; 
  } 
  
  c = cos( theta ) ; 
  s = sin( theta ) ; 
  
  if ( iaxis == 1 ) { 
    r[0] = 1.0 ; 
    r[4] = c ; 
    r[5] = s ; 
    r[7] = -s ; 
    r[8] = c ; 
  } 
  if ( iaxis == 2 ) { 
    r[0] = c ; 
    r[2] = -s ; 
    r[4] = 1.0 ; 
    r[6] = s ; 
    r[8] = c ; 
  } 
  if ( iaxis == 3 ) { 
    r[0] = c ; 
    r[1] = s ; 
    r[3] = -s ; 
    r[4] = c ; 
    r[8] = 1.0 ; 
  } 
} 

/* 
	Transformation from spherical angular variable, Xi, Eta ( radian ) 
		to rectangular variable, X, Y, Z 
		txexyz( xi, eta, x ) ; 
*/ 
void txexyz( double xi, double eta, double *x ){ 
  double eps = 1.0e-15 ; 

  if ( eta > M_PI / 2.0 ) { 
    if ( fabs( fabs( eta ) - M_PI / 2.0 ) <= eps ) 
      eta = M_PI / 2.0 ; 
  } 
  if ( eta < - M_PI / 2.0 ) { 
    if ( fabs( fabs( eta ) - M_PI / 2.0 ) <= eps ) 
      eta = - M_PI / 2.0 ; 
  } 
  if ( fabs ( eta ) < eps ) 
    eta = 0.0 ; 
  if ( fabs( eta ) > M_PI / 2.0 ) { 
    fprintf( stderr, "Error! Function txexyz --- Invalid value of angle eta!\n\teta=%20.15f\nAngle eta ( 2nd argument ) should be between -pi/2 and pi/2! \n", eta ) ; 
    printf( "eta = %20.15f \n", eta ) ; 
  } 
  
  x[0] = cos( eta ) * cos( xi ) ; 
  x[1] = cos( eta ) * sin( xi ) ; 
  x[2] = sin( eta ) ; 
} 

/* 
	Matrix multiplication  No. 1 
		vector y[NDIM] = matrix r[NDIM][NDIM] * vector x[NDIM] 
		mult0( r, x, y ) 
*/ 
void mult0( double *r, double *x, double *y ){ 
  int i, j ; 
  for ( i = 0 ; i < NDIM ; i++ ) { 
    y[i] = 0.0 ; 
    for ( j = 0 ; j < NDIM ; j++ ) 
      y[i] += r[NDIM*i+j] * x[j] ; 
  } 
}

// (alpha, delta) => (lambda, beta)
void eq2ec(double alpha, double delta, double *lambda, double *beta){
  // alpha;  right ascension [radian]
  // delta;  declination [radian]
  // lambda; ecliptic longitude [radian]
  // beta;   ecliptic latitude [radian]
  double ecp; // the mean obliquity of the ecliptic [radian]
  double t; // t = (year - 2000.0) / 100.0 )
  double t0, t1;
  double year;

  year = 2000.0;
  t = (year - 2000.0)/100.0;
  ecp = (84381.448/3600.) - (46.8150/3600.)*t - (0.00059/3600.)*t*t + (0.001813/3600.)*t*t*t; // [degree]
  ecp *= deg2rad; // [radian]
  *beta = asin( sin(delta)*cos(ecp) -cos(delta)*sin(alpha)*sin(ecp) );
  t0 = cos(delta)*cos(alpha);
  t1 = sin(delta)*sin(ecp) + cos(delta)*sin(alpha)*cos(ecp);
  t = atan2(t1, t0);
  if ( t < 0.0 ) t += 2*PI ;

  *lambda = t;
}

char *stringcp( char *s1, char *s2, int l ) {
/* almost the same as strncp() but '\0' is added to the end of the string */
  int i ; 
  for ( i = 0 ; i < l ; i++ ) 
    { 
      s1[i] = s2[i] ; 
      if ( s1[i] == '\0' ) return s1 ;  
    } 
  s1[l] = '\0' ; 
  return s1 ; 
}
