;+
; NAME: eis_make_status_fits.pro
;
; PURPOSE: To convert solar-B ccsds status packet archives into
;          level-0 fits format.
;          Converts ICU/PSU, CAM, MHC and Eis specific parameters of
;          HK1 status packets.
;      
; CATEGORY: Engineering
;
; CALLING SEQUENCE: eis_make_status_fits, 'status_archive' [,output=output]
;
; INPUTS: Filename string of the status archive
;
; KEYWORD PARAMETERS: Set output (string) to an existing directory
; where the fits output will be written. Default to '/tmp'.
;
; OUTPUTS: One fits file per status APID in either /tmp or the
; directory specified by output.
;
; CALLS: Various SSW fits routines.
;
; COMMON BLOCKS:None.
;
; PROCEDURE:
;
; RESTRICTIONS: Uses SolarSoft
;               To do: history keywords and updating
;
; MODIFICATION HISTORY:
;	17/09/06 mcrw   v2.0
;                       New version based on version 1.1
;                       Does not split ccsds header up
;       ??/11/06 mcrw   v2.1
;                       Added hk2 and aocs handling
;       28/01/07 mcrw   v2.2
;                       SOT_STS handling
;
; Needs altering so that the mhc mechanisms first/last position can be
; put in
;-

;
; Convert yyyymmddThhmmss to yyyy-mm-ddThh:mm:ss.000
;
function eis_convert_utc,ut_str
    yr_str = strmid(ut_str, 0, 4)
    mo_str = strmid(ut_str, 4, 2)
    da_str = strmid(ut_str, 6, 2)
    hr_str = strmid(ut_str, 9, 2)
    mi_str = strmid(ut_str, 11, 2)
    se_str = strmid(ut_str, 13, 2)
    return, yr_str+'-'+mo_str+'-'+da_str+'T'+hr_str+':'+mi_str+':'+se_str+'.000'
end


; Utility functions to extract bytes and bits.
; offset is the byte offset into the packet from location 0
; Extracts a byte from the array at offset location. Last 2 parameters
; unused but provided to make a standard calling sequence.
function eis_extractByte, dataArray, offset, startBit, numBits
    return, dataArray[offset, *]
end

; Extracts a 2-byte word
function eis_extractWord, dataArray, offset, startBit, numBits
    return, ishft(fix(dataArray[offset, *]), 8) or fix(dataArray[offset + 1, *])
end

; Extracts a 4-byte word
function eis_extractDoubleWord, dataArray, offset, startBit, numBits
    return, long(ishft(long(dataArray[offset,     *]), 24) or $
            ishft(long(dataArray[offset + 1, *]), 16) or $
            ishft(long(dataArray[offset + 2, *]),  8) or $
                  long(dataArray[offset + 3, *]))
end

; Extracts the top 4 bits from a byte
function eis_extractMsNibble, dataArray, offset, startBit, numBits
    return, byte(ishft(dataArray[offset, *], -4))
end

; Extracts the bottom 4 bits from a byte
function eis_extractLsNibble, dataArray, offset, startBit, numBits
    return, byte(dataArray[offset, *] and 15)
end

; Extracts n-bits from anywhere in a byte. No error checking.
; Bit order is: bit 0 (2^7) .. bit 7 (2^0), unfortunately
function eis_extractBitsFromByte, dataArray, offset, startBit, numBits
    foo = dataArray[offset, *]
    range = (startBit + numBits) - 1
    shift = 0 - (7 - range) ; negative for right shift
    mask = (2 ^ numBits) - 1
    return, byte(ishft(foo, shift) and mask)
end

; Extract n-bits from a 16 bit word. Starting offset is first byte
; of word
function eis_extractBitsFromWord, dataArray, offset, startBit, numBits
    word_array = eis_extractWord(dataArray, offset, startBit, numBits)
    range = (startBit + numBits) - 1
    shift = 0 - (15 - range) ; negative for right shift
    mask = (2 ^ numBits) - 1
    return, fix(ishft(word_array, shift) and mask)
end

function eis_extract24Bits, dataArray, offset, startBit, numBits
    long_array = eis_extractDoubleWord(dataArray, offset, startBit, numBits)
    return, long(ishft(long_array, -8) and '00FFFFFF'X)
end

;------------------------------------------------------------------------------------------

; Little test routines
;function eis_extractTcRecPktc, dataArray
;    return, extractWord(dataArray, 8)
;end

;function eis_extractEisMode, dataArray
;    return, extractMsNibble(dataArray, 1)
;end

;function eis_convertIcuCurrent, rawArray, polyArray
;    scale = 2.5 / 256
;    return, ((rawArray * scale) * polyArray[0] + polyArray[1])
;end

;------------------------------------------------------------------------------------------

; Add some standard keywords to the primary hdu.
; Spacecraft time comes from data
pro eis_add_primary_header_keywords, data, header, type, grt, sc_start, ut_start, sc_end, ut_end, ti_conv
;print, '--- : eis_add_primary_header_keywords'

    fxaddpar, header, 'origin', 'ISAS', 'Institute where file was written'
    fxaddpar, header, 'telescop', 'Hinode', 'Satellite name (Solar-B)'
    fxaddpar, header, 'instrume', 'EIS','EUV Imaging Spectrometer'
    fxaddpar, header, 'program', 'eis_make_status_fits','Program used to generate the file'
    fxaddpar, header, 'version', '2.0', 'Program version'
    fxaddpar, header, 'release', 'Preliminary', 'Confidence level'

    ; Use the sctime array and grt to add start and end
    ; times for the archive
    fxaddpar, header, 'gndtime', grt, 'Ground receive time'

    fxaddpar, header, 'ti0', string(sc_start, format='(Z08)'), 'Archive start time - Spacecraft time (TI)'
    fxaddpar, header, 'ti1', string(sc_end,format='(Z08)'), 'Archive end time - Spacecraft time (TI)'

    ut_start1 = eis_convert_utc(ut_start)
    ut_end1   = eis_convert_utc(ut_end)

;;;    yr_str = strmid(ut_start, 0, 8)
;;;    hr_str = strmid(ut_start, 9, 6)
;;;    ut_start1 = eis2cal(yr_str+hr_str)
;;;    yr_str = strmid(ut_end, 0, 8)
;;;    hr_str = strmid(ut_end, 9, 6)
;;;    ut_end1 = eis2cal(yr_str+hr_str)

;    fxaddpar, header, 'date_obs', ut_start1, 'Archive start time (UT)'
;    fxaddpar, header, 'date_end', ut_end1, 'Archive end time (UT)'

    fxaddpar, header, 'date_obs', ut_start1, 'Archive start time (UT)'
    fxaddpar, header, 'date_end', ut_end1, 'Archive end time (UT)'
    fxaddpar, header, 'ti_conv', ti_conv eq 1 ? 'T' : 'F', 'Conversion to UT success'
    ; Add packet frequency
    case type of
        'hk1'  : begin
            interval = 2
        end
        'hk2'  : begin
            interval = 2
        end
        'aocs1'  : begin
            interval = 1
        end
        'sts1' : begin
            interval = 2
        end
        'sts2' : begin
            interval = 10
        end
        'sts3' : begin
            interval = 10
        end
        'sot' : begin
            interval = 2
        end
        'eismdp' : begin
            interval = 2
        end
        else: interval = 0
    endcase
    fxaddpar, header, 'interval', interval, 'Packet frequencey (seconds)'

end

; Make a binary table extension for the spacecraft time and ccsds
; seequence counter for each APID type
;pro eis_mk_ccsds_bintable_extension, sctime, seqCount, fitsFile
pro eis_mk_ccsds_bintable_extension, stsData, fitsFile, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv
;    print, '--- Create bintable header for spacecraft'

    errmsg = ''
    fxbhmake, header, 1, 'SPACECRAFT', 'Ccsds Packet Information', errmsg=errmsg, /date, /initialize, /extver, /extlevel    
    if errmsg ne '' then print, '*** fxbhmake error: ' + errmsg

;    fxaddpar, header, 'UNIT', 'None', 'Units for parameters in this extension'

    eis_add_primary_header_keywords, stsData, header, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv

    sctime = eis_extractDoubleWord(stsData, 6, 0, 0)
    minv = min(sctime)
    maxv = max(sctime)

    ; Create the column description
    fxbaddcol, sc_col, header, sctime, $
      'TIME', 'MDP Time',              $
      tdmin = minv,                 $
      tdmax = maxv

    seqCount = eis_extractWord(stsData, 2, 0, 0)
    seqCount = seqCount and '3FFF'X
    minv = min(seqCount)
    maxv = max(seqCount)

    ; Create the column description
    fxbaddcol, seq_col, header, seqCount, $
      'SEQ_COUNTER', 'Ccsds Sequence Counter',              $
      tdmin = minv,                 $
      tdmax = maxv

    ; Create the fits header
    errmsg = ''
    fxbcreate, alun, fitsFile, header, errmsg = errmsg
    if errmsg ne '' then print, '*** fxbcreate error: ' + errmsg

    ; Write out the data
    errmsg = ''
    fxbwrite, alun, sctime, sc_col, 1, errmsg=errmsg
    if errmsg ne '' then print, '*** fxbwrite error: ' + errmsg

    errmsg = ''
    fxbwrite, alun, seqCount, seq_col, 1, errmsg=errmsg
    if errmsg ne '' then print, '*** fxbwrite error: ' + errmsg

    ; Finish up
    errmsg = ''
    fxbfinish, alun, errmsg=errmsg
    if errmsg ne '' then print, 'Error: ' + errmsg

; Debug
;print
;print, 'HEADER:'
;print, header

end

; Make bintable extensions for the MHC subcomm table - unfinished.
pro eis_mk_subcomm_extension, stsData, paramFile, fitsFile
return
;    print, '--- Create bintable header for subcomm table'

    errmsg = ''
    fxbhmake, header, 1, 'SUBCOMM VALUES', 'Subcomm table elements', errmsg=errmsg, /date, /initialize, /extver, /extlevel    
    if errmsg ne '' then print, '*** fxbhmake error: ' + errmsg

    fxaddpar, header, 'UNIT', 'None', 'Units for parameters in this extension'

    minv = min(sctime)
    maxv = max(sctime)

    ; Create the column description
    fxbaddcol, sc_col, header, sctime, $
      'TIME', 'MDP Time',              $
      tdmin = minv,                 $
      tdmax = maxv

    minv = min(seqCount)
    maxv = max(seqCount)

    ; Create the column description
    fxbaddcol, seq_col, header, seqCount, $
      'SEQ_COUNTER', 'Ccsds Sequence Counter',              $
      tdmin = minv,                 $
      tdmax = maxv

    ; Create the fits header
    errmsg = ''
    fxbcreate, alun, fitsFile, header, errmsg = errmsg
    if errmsg ne '' then print, '*** fxbcreate error: ' + errmsg

    ; Write out the data
    errmsg = ''
    fxbwrite, alun, sctime, sc_col, 1, errmsg=errmsg
    if errmsg ne '' then print, '*** fxbwrite error: ' + errmsg

    errmsg = ''
    fxbwrite, alun, seqCount, seq_col, 1, errmsg=errmsg
    if errmsg ne '' then print, '*** fxbwrite error: ' + errmsg

    ; Finish up
    errmsg = ''
    fxbfinish, alun, errmsg=errmsg
    if errmsg ne '' then print, 'Error: ' + errmsg

end

; Make a binary table extension, complete with data
;;;pro eis_mk_bintable_extension, data, parameterFile, fitsFile
pro eis_mk_bintable_extension, data, parameterFile, fitsFile, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv
;print, '--- eis_mk_simple_bintable_extension'

    ; Open the parameter file which contains the fits definitions
    openr, lun, parameterFile, /get_lun

    ; Read in extension header details
    extName = ''
    extComment = ''
    numParams = 0B
    unit = ''
    desc = ''
    cuni = ''
    readf, lun, extName
    readf, lun, extComment
    readf, lun, numParams
;    readf, lun, unit
;    readf, lun, desc
;    readf, lun, cuni

    num_readings = n_elements(data[0,*])
    ; Create the parameter objects and read in their details
    param = objarr(numParams)
    for i = 0, numParams - 1 do begin
        obj = obj_new('eis_status_parameter')
        obj->read, lun
        obj->setData,data, num_readings
;;;        obj->describe ; Debug
        param[i] = obj
    endfor

    ; Close parameter file
    close, lun
    free_lun, lun

    ; Make the bintable extension header
;    print, '--- Create bintable header for ' + extName
    errmsg = ''
    fxbhmake, header, 1, extName, extComment, errmsg=errmsg, /date, /initialize, /extver, /extlevel
    if errmsg ne '' then print, '*** fxbhmake error: ' + errmsg
;;;    fxaddpar, header, 'UNIT', unit, 'Units for parameters in this extension'

    eis_add_primary_header_keywords, data, header, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv


    ; Create the column description
    for i = 0, numParams - 1 do begin
        fxbaddcol, col, header, param[i]->data(), $
          param[i]->name(), param[i]->comment(),  $
          tunit = param[i]->unit(),               $
          tdmin = param[i]->minval(),             $
          tdmax = param[i]->maxval()
    endfor

    ; Create the fits header
    errmsg = ''
    fxbcreate, alun, fitsFile, header, errmsg = errmsg
    if errmsg ne '' then print, '*** fxbcreate error: ' + errmsg

    ; col is an output really?!?! *** Check fxbwrite ***
    ; Write out the data
    errmsg = ''
    for col = 1, numParams do begin
        fxbwrite, alun, param[col - 1]->data(), col, 1, errmsg=errmsg
    endfor
    if errmsg ne '' then print, '*** fxbwrite error: ' + errmsg

    ; Finish up, fxbfinish disposes of alun
    errmsg = ''
    fxbfinish, alun, errmsg=errmsg
    if errmsg ne '' then print, '*** fxbfinish error: ' + errmsg

    ; Get rid of the objects
    obj_destroy,param
end

; General procedure for writing out the fits extensions
pro eis_mkStsFits, file, stsData, fitsBase, parameterBase, type, pfiles

    ; File will have a filename:
    ;     eis_sts_yyyymmdd_hhmmssttmm
    ; and will create a fits file with name:
    ;     eis_sts1_yyymmdd_hhmmss.fits
    ; for a status 1 archive
    ; hk1, hk2, hk3, sts1, sts2, sts3

    ; Do spacecraft time conversions here to send to add_primary_header
    ; Get the first scpacecraft time (ti)
    sc_time = (eis_extractDoubleWord(stsData, 6, 0, 0))[0]
    ;
    ; fudge for mdp time cock-up
    ;
;    sc_time = sc_time or '30000000'X
    
;    print, 'sc_time = ', string(sc_time, format='(Z)')

    last_sample = n_elements(stsData[0,*]) - 1
;print, 'last_sample = ', last_sample
    sc_time_end = (eis_extractDoubleWord(stsData, 6, 0, 0))[last_sample]
    ;
    ; fudge for mdp time cock-up
    ;
;    sc_time_end = sc_time_end or '30000000'X
;    print, 'sc_time_end = ', string(sc_time_end, format='(Z)')

    ; Create the fits file filename
    break_file, file, disk_log, dir, filnam, ext, fversion, node

    yr_str = strmid(filnam, 8, 8)
    hr_str = strmid(filnam, 17, 6)
    grt = yr_str + hr_str
;;;;    ut_str = eis_ti2utc(string(sc_time), grt)
    ut_str = eis_ti2utc(sc_time, grt)

;;;;    ut_end   = eis_ti2utc(string(sc_time_end), yr_str + hr_str)
    ut_end   = eis_ti2utc(sc_time_end, yr_str + hr_str)

    converted_ok = strpos(ut_str, 'T', 8)
    if converted_ok ne -1 then begin ; Good conversion
        new_ut_str = ut_str
;;;        strput,ut_str,'_',8
        strput,new_ut_str,'_',8
        ; Make fits file name from ut string
        fitsFile = fitsBase + 'eis_' + type + '_' + new_ut_str + '.fits'
        ; More processing for header (convered ok etc)
        ti_conv = 1
    endif else begin
        ; Make fits file name from grt string
        fitsFile = fitsBase + 'eis_' + type + '_' + yr_str + '_' + hr_str + '.fits'
        ; More processing for header, bad case
        ti_conv = 0
    endelse

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;    fitsFile = fitsBase + 'eis_' + type + '_' + ut_str + '.fits'

    ; Make the primary header
;    print, 'eis_mkStsFits: making FITS header for type ' + type ; Debug
    fxhmake, header, /initialize, /date, /extend

    ; Add primary header keywords
    eis_add_primary_header_keywords, stsData, header, type, grt, sc_time, ut_str, sc_time_end, ut_end, ti_conv
    
    ; Write the header to file
    fxwrite, fitsFile, header

    ; Loop through the parameter filename array writing out the extensions
    for i = 0, n_elements(pfiles) - 1 do begin
;;;        eis_mk_bintable_extension, stsData, parameterBase + pfiles[i], fitsFile
        eis_mk_bintable_extension, stsData, parameterBase + pfiles[i], fitsFile, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv
    endfor

    ; Not done yet
;    if keyword_set(sub) then $
;      eis_mk_subcom_extension, stsData, parameterBase+'/mhc_subcom_vals', fitsFile

    ; Chuck in the spacecraft parameters
;;;    eis_mk_ccsds_bintable_extension, sctime, seqCountArray, fitsFile
    eis_mk_ccsds_bintable_extension, stsData, fitsFile, type, grt, sc_time, ut_str, sc_time_end, ut_end,ti_conv
end

; Make the fits file for the icu/psu APID
pro eis_mkIcuPsuFits, file, sts1Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(11)
    parameterFilenames[0]  = '/psu_voltages'
    parameterFilenames[1]  = '/psu_currents'
    parameterFilenames[2]  = '/psu_temperatures'
    parameterFilenames[3]  = '/icu_misc'
    parameterFilenames[4]  = '/icu_status'
    parameterFilenames[5]  = '/psu_status'
    parameterFilenames[6]  = '/hm_status'
    parameterFilenames[7]  = '/icu_interfaces'
    parameterFilenames[8]  = '/icu_errors'
    parameterFilenames[9]  = '/icu_counters'
    parameterFilenames[10] = '/icu_science'

    eis_mkStsFits, file, sts1Data, fitsBase, parameterBase, 'sts1', parameterFilenames
end

; Make the fits file for the cam APID
pro eis_mkCamFits, file, sts2Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(8)
    parameterFilenames[0] = '/cam_voltages'
    parameterFilenames[1] = '/cam_currents'
    parameterFilenames[2] = '/cam_biases'
    parameterFilenames[3] = '/cam_psu_voltages'
    parameterFilenames[4] = '/cam_psu_currents'
    parameterFilenames[5] = '/cam_icu_parameters'
    parameterFilenames[6] = '/cam_misc'
    parameterFilenames[7] = '/cam_temperatures'

    eis_mkStsFits, file, sts2Data, fitsBase, parameterBase, 'sts2', parameterFilenames
end

; Make the fits file for the mhc APID
pro eis_mkMhcFits, file, sts3Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(10)
    parameterFilenames[0] = '/mhc_references'
    parameterFilenames[1] = '/mhc_voltages'
    parameterFilenames[2] = '/mhc_currents'
    parameterFilenames[3] = '/mhc_temperatures_gp0'
    parameterFilenames[4] = '/mhc_temperatures_gp2'
    parameterFilenames[5] = '/mhc_mechanisms'
    parameterFilenames[6] = '/mhc_status'
    parameterFilenames[7] = '/mhc_counters'
    parameterFilenames[8] = '/mhc_qcm'
    parameterFilenames[9] = '/mhc_subcom'

    eis_mkStsFits, file, sts3Data, fitsBase, parameterBase, 'sts3', parameterFilenames
end

; Make the fits file for the hk1 APID
pro eis_mkHk1Fits, file, hk1Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(3)
    parameterFilenames[0] = '/hk1_temperatures'
    parameterFilenames[1] = '/hk1_status'
    parameterFilenames[2] = '/hk1_mdp'

    eis_mkStsFits, file, hk1Data, fitsBase, parameterBase, 'hk1', parameterFilenames
end

; Make the fits file for the hk2 APID
pro eis_mkHk2Fits, file, hk2Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(1)
    parameterFilenames[0] = '/hk2_ufs'

    eis_mkStsFits, file, hk2Data, fitsBase, parameterBase, 'hk2', parameterFilenames
end

; Make the fits file for the hk aocs pro 1 APID (440)
pro eis_mkHkAocs1Fits, file, hkAocs1Data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(1)
    parameterFilenames[0] = '/hk_acu'

    eis_mkStsFits, file, hkAocs1Data, fitsBase, parameterBase, 'aocs1', parameterFilenames
end

; Make the fits file for the sot sts data APID (0x542)
pro eis_mk_sot_sts_fits, file, sot_data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(1)
    parameterFilenames[0] = '/sot_sts'

    eis_mkStsFits, file, sot_data, fitsBase, parameterBase, 'sot', parameterFilenames
end

pro eis_mk_eis_mdp_fits, file, eis_mdp_data, fitsBase, parameterBase

    ; Initialise the parameter filename array and basename string
    parameterFilenames = strarr(1)
    parameterFilenames[0] = '/eis_mdp'

    eis_mkStsFits, file, eis_mdp_data, fitsBase, parameterBase, 'eismdp', parameterFilenames
end

;------------------------------------------------------------------------------------------

; Main procedure
pro eis_make_status_fits, file, output=output

    openr, lu, file, /get_lun

    ; Extract gnd receive time from filename
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;    grt = '20060811113116' ; Dummy for testing

    ; Initalize packet counters
    packetCount   = 0L
    sts1Count     = 0L
    sts2Count     = 0L
    sts3Count     = 0L
    hk1Count      = 0L
    hk2Count      = 0L
    hk440Count    = 0L
    sot_sts_count = 0L
    eis_mdp_count = 0L

    ; Initialize start times
;    sts1StartTime   = 0L
;    sts2StartTime   = 0L
;    sts3StartTime   = 0L
;    hk1StartTime    = 0L
;    hk2StartTime    = 0L
;    hk440StartTime  = 0L
;    sot_sts_start_time = 0L

    ; Initialize flags that control appending data onto the byte arrays
    sts1Flag   = 0
    sts2Flag   = 0
    sts3Flag   = 0
    hk1Flag    = 0
    hk2Flag    = 0
    hk440Flag  = 0
    sot_sts_flag = 0
    eis_mdp_flag = 0

    ; Allocate room for 10,000 packets of each APID type (a tad excessive perhaps)
    sts1Data   = bytarr(110L * 10000, /nozero)
    sts2Data   = bytarr(160L * 10000, /nozero)
    sts3Data   = bytarr(160L * 10000, /nozero)
    hk1Data    = bytarr(436L * 10000, /nozero)
    hk2Data    = bytarr(436L * 10000, /nozero)
    hk440Data  = bytarr(436L * 10000, /nozero)
    sot_sts_data = bytarr(208L * 10000, /nozero)
    eis_mdp_data = bytarr(29L * 10000, /nozero)

    ; Ccsds packet stuff
    ccsdsHeader = bytarr(10)	; Split so we can get ccsds seq count (not in hk1)

;    ccsdsSequenceCount = 0L
;    spacecraftTime     = 0L

    ; Array for the packet data of each apid
    sts1Packet  = bytarr(100)
    sts2Packet  = bytarr(150)
    sts3Packet  = bytarr(150)
    hk1Packet   = bytarr(426)
    sot_sts_packet = bytarr(198)
    eis_mdp_packet = bytarr(19)

    on_ioerror, FINISH
    
    ; Read in the data...
    while not eof(lu) do begin

        ; Get the Ccsds headers
        readu, lu, ccsdsHeader
;        readu, lu, spacecraftTime

;        temp = ccsdsHeader[2:3]
;        ccsdsSequenceCount = ishft(fix(temp[0]), 8) + temp[1]
;        ccsdsSequenceCount = ccsdsSequenceCount and '3FFF'X

;        byteorder, spacecraftTime, /htonl

        ; Calculate APID and user data length (used to create buffers on the fly)
        apid = ishft((ulong(ccsdsHeader[0]) and 7), 8) or ulong(ccsdsHeader[1])

        ; Process each type of packet according to its APID
        case apid of

	    '420'X : begin	;;; hk1 (only in ISAS generated archives)
                readu, lu, hk1Packet
                if hk1Flag eq 0 then begin
                    hk1Flag = 1
                    hk1Data = [ccsdsHeader, hk1Packet]
                endif else begin
                    hk1Data = [hk1Data, [ccsdsHeader, hk1Packet]]
                endelse
                hk1Count = hk1Count + 1
                packetCount = packetCount + 1
            end
	    '428'X : begin	;;; (only in ISAS generated archives)
                readu, lu, hk1Packet
                if hk2Flag eq 0 then begin
                    hk2Flag = 1
                    hk2Data = [ccsdsHeader, hk1Packet]
                endif else begin
                    hk2Data = [hk2Data, [ccsdsHeader, hk1Packet]]
                endelse
                hk2Count = hk2Count + 1
                packetCount = packetCount + 1
            end
	    '440'X : begin	;;; (only in ISAS generated archives)
                readu, lu, hk1Packet
                if hk440Flag eq 0 then begin
                    hk440Flag = 1
                    hk440Data = [ccsdsHeader, hk1Packet]
                endif else begin
                    hk440Data = [hk440Data, [ccsdsHeader, hk1Packet]]
                endelse
                hk440Count = hk440Count + 1
                packetCount = packetCount + 1
            end
            '5C4'X : begin	;;; Status 1
                readu, lu, sts1Packet
                if sts1Flag eq 0 then begin
                    sts1Flag = 1
                    sts1Data = [ccsdsHeader, sts1Packet]
                endif else begin
                    sts1Data = [sts1Data, [ccsdsHeader, sts1Packet]]
                endelse
                sts1Count = sts1Count + 1
                packetCount = packetCount + 1
            end
            '5C6'X : begin	;;; Status 2
                readu, lu, sts2Packet
                if sts2Flag eq 0 then begin
                    sts2Flag = 1
                    sts2Data = [ccsdsHeader, sts2Packet]
                endif else begin
                    sts2Data = [sts2Data, [ccsdsHeader, sts2Packet]]
                endelse
                sts2Count = sts2Count + 1
                packetCount = packetCount + 1
            end
            '5C8'X : begin	;;; Status 3
                readu, lu, sts3Packet
                if sts3Flag eq 0 then begin
                    sts3Flag = 1
                    sts3Data = [ccsdsHeader, sts3Packet]
                endif else begin
                    sts3Data = [sts3Data, [ccsdsHeader, sts3Packet]]
                endelse
                sts3Count = sts3Count + 1
                packetCount = packetCount + 1
            end
            '542'X : begin	;;; sot_sts
                readu, lu, sot_sts_packet
                if sot_sts_flag eq 0 then begin
                    sot_sts_flag = 1
                    sot_sts_data = [ccsdsHeader, sot_sts_packet]
                endif else begin
                    sot_sts_data = [sot_sts_data, [ccsdsHeader, sot_sts_packet]]
                endelse
                sot_sts_count = sot_sts_count + 1
                packetCount = packetCount + 1
            end
            '5C2'X : begin	;;; eis_mdp
                readu, lu, eis_mdp_packet
                if eis_mdp_flag eq 0 then begin
                    eis_mdp_flag = 1
                    eis_mdp_data = [ccsdsHeader, eis_mdp_packet]
                endif else begin
                    eis_mdp_data = [eis_mdp_data, [ccsdsHeader, eis_mdp_packet]]
                endelse
                eis_mdp_count = eis_mdp_count + 1
                packetCount = packetCount + 1
            end
;	    '0'X  : begin	;;; loss of hk detected by egse (only in MSSL generated archives)
;                len  = ishft(ulong(ccsdsHeader[4]), 8) or ulong(ccsdsHeader[5]) - 3
;                buffer = bytarr(len)
;                readu, lu, buffer
;            end
;	    'F0'X  : begin	;;; start of archive (only in MSSL generated archives)
;                len  = ishft(ulong(ccsdsHeader[4]), 8) or ulong(ccsdsHeader[5]) - 3
;                buffer = bytarr(len)
;                readu, lu, buffer
;            end
;	    '7FF'X  : begin	;;; start of hk (only in MSSL generated archives)
;                len  = ishft(ulong(ccsdsHeader[4]), 8) or ulong(ccsdsHeader[5]) - 3
;                buffer = bytarr(len)
;                readu, lu, buffer
;            end
;            '1CC'X : begin	;;; Telecommand (only in MSSL generated archives)
;                len  = ishft(ulong(ccsdsHeader[4]), 8) or ulong(ccsdsHeader[5]) - 3
;                buffer = bytarr(len)
;                readu, lu, buffer
;            end
            else : begin
                print, 'Unknown APID ', apid ; Bit of an error if this happens...
            end
        endcase
    endwhile

FINISH:

    ; All data read in - debug print
;    print, 'Packet Count : ', packetCount
;    print, 'HK 1 packets : ', hk1Count
;    print, 'HK 2 packets : ', hk2Count
;    print, 'HK 440 packets : ', hk440Count
;    print, 'Status 1 packets : ', sts1Count
;    print, 'Status 2 packets : ', sts2Count
;    print, 'Status 3 packets : ', sts3Count

;help,hk1Data
;help,sts1Data
;help,sts2Data
;help,sts3Data

    ; Close the input file
    close, lu
    free_lun, lu

    ; No point continuing if there are no packets
    if packetCount eq 0 then return

    ; Re-arrange the data into columns, if there is any data that is...
    if sts1Count gt 0 then begin
        sts1Data = reform(temporary(sts1Data), 110, sts1Count, /overwrite)
    endif
    if sts2Count gt 0 then begin
        sts2Data = reform(temporary(sts2Data), 160, sts2Count, /overwrite)
    endif
    if sts3Count gt 0 then begin
        sts3Data = reform(temporary(sts3Data), 160, sts3Count, /overwrite)
    endif

    ; In case we are dealing with a MSSL archive which does not have hk1
    if hk1count gt 0 then begin
        hk1Data  = reform(temporary(hk1Data), 436, hk1Count, /overwrite)
    endif

    if hk2count gt 0 then begin
        hk2Data  = reform(temporary(hk2Data), 436, hk2Count, /overwrite)
    endif

    if hk440count gt 0 then begin
        hk440Data  = reform(temporary(hk440Data), 436, hk440Count, /overwrite)
    endif
    
    if sot_sts_count gt 0 then begin
        sot_sts_data = reform(temporary(sot_sts_data), 208, sot_sts_count, /overwrite)
    endif

    if eis_mdp_count gt 0 then begin
        eis_mdp_data = reform(temporary(eis_mdp_data), 29, eis_mdp_count, /overwrite)
    endif

    ; Make a fits file for each APID type...
    if keyword_set(output) then begin
        fitsbase = output
    end else begin
        fitsBase = '/tmp/' ; Where to write the fits file
    endelse

;;;    parameterBase = '/home/matt/idl/fits/keywords/new/' ; Where to find the keywords
;    parameterBase = '$EIS_STATUS_FITS_KEYWORDS' ; Where to find the keywords

;;;;;;;;;;;;;;;;;;;    parameterBase = getenv('EIS_STATUS_FITS_KEYWORDS') ; Where to find the keywords

    parameterBase = '$HOME/work/idl/keywords/'
;print,'USING ',parameterBase

;help,hk1Data
;help,sts1Data
;help,sts2Data
;help,sts3Data

;print, sts1Data[12:13,*]

;return

    if sts1Count gt 0 then eis_mkIcuPsuFits, file, sts1Data,   fitsBase, parameterBase
    if sts2Count gt 0 then eis_mkCamFits,    file, sts2Data,   fitsBase, parameterBase
    if sts3Count gt 0 then eis_mkMhcFits,    file, sts3Data,   fitsBase, parameterBase
    if hk1count  gt 0 then eis_mkHk1Fits,    file, hk1Data,    fitsBase, parameterBase
    if hk2count  gt 0 then eis_mkHk2Fits,    file, hk2Data,    fitsBase, parameterBase
    if hk440count  gt 0 then eis_mkHkAocs1Fits,file, hk440Data,  fitsBase, parameterBase
    if sot_sts_count gt 0 then eis_mk_sot_sts_fits, file, sot_sts_data, fitsBase, parameterBase
    if eis_mdp_count gt 0 then eis_mk_eis_mdp_fits, file, eis_mdp_data, fitsBase, parameterBase

end
