;+
; NAME: eis_status_parameter__define.pro
;
; PURPOSE: To create an object that contains information defining
; 	   individual status parameters for EIS.
;      
; CATEGORY: Engineering
;
; CALLING SEQUENCE: None
;
; INPUTS: Lun of a file containing parameter descriptions.
;
; KEYWORD PARAMETERS: None
;
; OUTPUTS: None
;
; CALLS: None
;
; COMMON BLOCKS:None.
;
; PROCEDURE:
;
; RESTRICTIONS: None
;
; MODIFICATION HISTORY:
;	23/11/05 1.0 mcrw wrote
;       02/02/07 1.1 mcrw removed maskign for 16 bit data
;-

; Method to read the parameter details from the parameter definition file.
pro eis_status_parameter::read, lu
    str=''
    readf, lu, str            ; Parameter name
    self.name = str
    readf, lu, str            ; Parameter description
    self.comment = str
    a = 0L
    b = 0B
    c = 0B
    readf, lu, a              ; Word size
    self.size = a
    readf, lu, a, b, c, str   ; Offset, start bit, bit width
    self.offset = a
    self.startBit = b
    self.numBits = c
    self.func = str           ; Name of extraction function
    readf, lu, str
    self.unit = str
end

; Accessor method to set the parameter's data array. Data is the
; entire packet archive for this unit (icu, cam, etc). Calls the
; function in self.func to extract the data.
pro eis_status_parameter::setData, data, numRows
    case self.size of
        1 : self.data = ptr_new(bytarr(numRows))  ; 8  bits
        2 : self.data = ptr_new(uintarr(numRows)) ; 16 bits
        3 : self.data = ptr_new(lonarr(numRows))  ; 24 bits
        4 : self.data = ptr_new(lonarr(numRows))  ; 32 bits
    endcase
    dat = call_function(strtrim(self.func,2), data, self.offset, self.startBit, self.numBits)
    *self.data = reform(dat, numRows)
    if self.size eq 3 then begin
        ; This will wipe out the sign bit for signed parameters
        *self.data = (*self.data and '00FFFFFF'x)
        ; To restore the sign bit:
;        *self.data = ishft(*self.data, -4)
;        *self.data = *self.data/128
    endif
    if self.size eq 2 then begin
;        print, 'Masking (2) ', self.name
;        *self.data = (*self.data and '0000FFFF'x)
        *self.data = (*self.data and 'FFFF'x)
    endif
end

; Accessor method to return the parameter's data array
function eis_status_parameter::data
    return, *self.data
end

; Accessor method to return parameter's unit
function eis_status_parameter::unit
    return, self.unit
end

; Accessor method to return parameter's minimum value data as ulong
function eis_status_parameter::minval
;    case self.size of
;        1 : return, byte(min(*self.data))
;        2 : return, uint(min(*self.data))
;        3 : return, ulong(min(*self.data))
;        4 : return, ulong(min(*self.data))
;    endcase
;;;    return, min(*self.data)
    return, ulong(min(*self.data))
end

; Accessor method to return parameter's maximum value data as ulong
function eis_status_parameter::maxval
;    case self.size of
;        1 : return, byte(max(*self.data))
;        2 : return, uint(max(*self.data))
;        3 : return, ulong(max(*self.data))
;        4 : return, ulong(max(*self.data))
;    endcase
;;;    return, max(*self.data)    
    return, ulong(max(*self.data))    
end

; Accessor method to return parameter name
function eis_status_parameter::name
    return, self.name 
end

; Accessor method to return parameter comment
function eis_status_parameter::comment
    return, self.comment
end

; Debugging method to print parameter details
pro eis_status_parameter::describe
    print, 'name : ', self.name
    print, 'size : ', self.size
    print, 'func : ', self.func
    print, 'offset : ', self.offset
    print, 'startBit : ', self.startBit
    print, 'numBits : ', self.numBits
    print, 'unit : ', self.unit
end

; The object which contains all the information needed to extract a
; parameter from the status archive file.
pro eis_status_parameter__define

    struct = { eis_status_parameter, $
               name	: '', $ ; Of parameter in fits file
               offset	: 0L, $ ; Offset, in bytes, from start of packet
               func	: '', $ ; Function number to extract parameter
               startBit	: 0B, $ ; Startbit in offset byte
               numBits	: 0B, $ ; Bit width of parameter
               comment	: '', $ ; String describing parameter to go into FITS file
               size	: 0B, $ ; Byte size of parameter
               unit     : '', $ ; Unit of parameter
               data	: ptr_new()} ; Data array

end

