package Data;

#print "HINODE/EIS/EISPIPELINE/EISMISSION/DATA.PM\n";

use strict;
use warnings;

use File::Basename;

use Trace::Trace;

use Hinode::EIS::EisPipeline::Dates;
use Hinode::EIS::EisPipeline::Filesystem;
use Hinode::EIS::EisPipeline::Flags;
use Hinode::EIS::EisPipeline::Pipeline;
use Hinode::EIS::EisPipeline::Program;
use Hinode::EIS::EisPipeline::System;
use Hinode::EIS::EisPipeline::EisMission::Ccsds;
use Hinode::EIS::EisPipeline::EisMission::Filesystem;
use Hinode::EIS::EisPipeline::EisMission::Mission;
use Hinode::EIS::EisPipeline::EisMission::Program;

sub BEGIN {
    print STDOUT "BEGIN Hinode/EIS/EisPipeline/EisMission/Data\n";
}

=head3 Private module variables (my)                                                                                    
=cut                                                                                                                    

=begin html                                                                                                             
                                                                                                                        
<ul>                                                                                                                    
<li>@received_files                                                                                                     
A list holding all the names of the files received, in order of reception.                                              
</li>                                                                                                                   
<li>@joined_files                                                                                                       
A list of all the filenames after the join process has finished.                                                        
</li>                                                                                                                   
<li>@damaged_files                                                                                                      
A list of the files which are not complete.                                                                             
</li>                                                                                                                   
</ul>                                                                                                                   
                                                                                                                        
=end html                                                                                                               
                                                                                                                        
=cut
my @received_files = ();
my @joined_files = ();
my @damaged_files = ();

=head3 Publicly available module variables (our)                                                                        
=cut                                                                                                                    

=begin html                                                                                                             
                                                                                                                        
<table>                                                                                                                 
    <tr>                                                                                                                
    <td>$translate_log</td>                                                                                             
    <td>Ascii text file containing the translation log.</td>                                                            
    </tr>                                                                                                               
    <td>$translate_rescue_log</td>                                                                                      
    <td>The ascii error log.</td>                                                                                       
    </tr>                                                                                                               
    <td>%translate</td>                                                                                                 
    <td>Hash containg the reanslation from md files to fits files.</td>                                                 
    </tr>                                                                                                               
    <td>@empty_files</td>                                                                                               
    <td>List of files which are empty.</td>                                                                             
    </tr>                                                                                                               
    <tr>                                                                                                                
    <td>@non_compressed_files</td>                                                                                      
    <td>List of files which are not compressed</td>                                                                     
    </tr>                                                                                                               
    <td>$number_pre_decompression</td>                                                                                  
    <td>Number of files before decompression.</td>                                                                      
    </tr>                                                                                                               
    <tr>                                                                                                                
    <td>$number_post_decompression</td>                                                                                 
    <td>Number of files after decompression.</td>                                                                       
    </tr>                                                                                                               
</table>                                                                                                                
                                                                                                                        
=end html                                                                                                               
                                                                                                                        
=cut
our $translate_log = "";
our $translate_rescue_log = "";
our %translate = ();
our @empty_files = ();
our @non_compressed_files = ();
our $number_pre_decompression = 0;
our $number_post_decompression = 0;

=head3 Functions exported for global access                                                                             
                                                                                                                        
=cut                                                                                                                    

sub init_logs {
    Trace::trace((caller(0))[3]);

#    $darts_md_check_log = "$Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/md_hdrs.$Dates::start_date_string.html";
    $Filesystem::DARTS_MD_CHECK_LOG = "$Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/md_hdrs.$Dates::start_date_string.html";

    #    $decompression_log = $Filesystem::LOCAL_DECOMPRESSION_LOG . "$Dates::start_date_string" . ".txt";
#    $reformat_log = $Filesystem::LOCAL_MD_REFORMAT_LOG . "$Dates::start_date_string" . ".txt";    
}

# Check for the existence of a list containing the ccsds files which should not be reformatted as they crash the reformatter...
sub hooligan_check {
    &$System::comment("Checking for bad input ccsds files in $Filesystem::HOOLIGANS_DIR");
    if (open(FD, "< $Filesystem::NO_REFORMAT_FILE")) {
        while (<FD>) {
            chomp;
            &$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::HOOLIGANS");
	}
        close FD;
        &$System::perform("$Program::UNLINK $Filesystem::NO_REFORMAT_FILE");
        &$System::perform("$Program::TOUCH  $Filesystem::NO_REFORMAT_FILE");
    }
    else {
        &$System::log("Unable to read $Filesystem::NO_REFORMAT_FILE");
    }
}

=head3 Private module functions                                                                                         
                                                                                                                        
=cut                                                                                                                    

sub create_decompress_idl_program {
    Trace::trace((caller(0))[3]);
    my $output = "!quiet=1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_ccsds_interface__define\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp_script\n";
    $output .= ";eis_md_decomp_script,'$Filesystem::JOIN_DIR', /merge\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/mdppmdctrecover\n";
    $output .= ";;;.comp $Filesystem::LOCAL_IDL/mdppmdctrecover_merge\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp_fix\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp_fix_script\n";
    $output .= ";eis_md_decomp_fix_script,'$Filesystem::JOIN_DIR', /merge\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_fix1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_fix1_script\n";
    $output .= "eis_md_decomp_fix1_script, '$Filesystem::MERGE_DIR', /merge\n";
    $output .= "!quiet=0\n";

    # mcrw 20130903 - write to fixed path
    System::dprint("Writing to $Program::MD_DECOMP");
    &$System::comment("Writing to $Program::MD_DECOMP");
    open(MD_DECOMP, "> $Program::MD_DECOMP");
    print MD_DECOMP $output;
    close MD_DECOMP;
    
    &$System::log("IDL file:\n$output");
}

sub create_reformat_idl_program {
    my $reformat_log = shift;
    my $old_seq_str = shift;

    Trace::trace((caller(0))[3]);
    my $output = "!quiet=0\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/db_read_sciencestudyraster_entry\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_read_orbit_events_reformatter\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_hdr__define\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_data__readfits\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_plan__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_cal__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_data__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_data__readccsds\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_fits_calib\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_fits_coord\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_fits_dataid\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_spcd__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_fits_script\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_fits_obstime\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_mkfits\n";
    $output .= ";!quiet=1\n";
    $output .= "eis_fits_script,'$Filesystem::DECOMPRESSED_DIR', '$Filesystem::TEMP_FITS_DIR/',logfile='$Date::translate_log' $old_seq_str\n";
    $output .= ";eis_fits_script,'$Filesystem::DECOMPRESSED_DIR', '$Filesystem::TEMP_FITS_DIR/',logfile='$reformat_log', /noplan\n";
    $output .= ";eis_fits_script,'$Filesystem::DECOMPRESSED_DIR', '$Filesystem::TEMP_FITS_DIR/',logfile='$reformat_log', /nospcd\n";

    &$System::stage_step("Writing to $Program::MD_REFORMAT");
    open(MD_FITS, "> $Program::MD_REFORMAT");
    print MD_FITS $output;
    close MD_FITS;
    
    &$System::comment("IDL file:\n$output");
}

sub old_rescue {
    Trace::trace((caller(0))[3]);
    &$System::stage_step("Run the old recover program over each file");
    &$System::comment("Old rescue");
}

# rescue_chk_tbl.sh -m
sub jpeg_rescue {
    Trace::trace((caller(0))[3]);
    &$System::stage_step("jpeg_rescue: run the new jpeg recover program over each file");
    &$System::comment("Get the files in $Filesystem::NURSERY_DIR");
    my @damaged_files = &$System::list($Filesystem::NURSERY_DIR, "eis_md*");
    foreach my $file (@damaged_files) {
	chomp $file;
	my $f = basename($file);
###	&$System::perform("$Program::RESCUE $file $System::NURSERY_RESCUED_DIR/$f");
	&$System::perform("$Program::NEW_RESCUE $file $Filesystem::NURSERY_RESCUED_DIR/$f");
    }
}

sub perform_headless_check {
    Trace::trace((caller(0))[3]);
    &$System::stage_step("Check which mission data files have missing md headers and move them to $Filesystem::NURSERY_DIR");
    if (-e "$Filesystem::HEADLESS_PACKETS_LOG") {
	&$System::comment("Headless packets file exists");
	open(BAD, "< $Filesystem::HEADLESS_PACKETS_LOG");
	while (<BAD>) {
	    next if /^Headless/;
	    last if /^$/;
	    chomp;
	    if ($Flags::RESCUE_MOD) {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::NURSERY_DIR") if (-e "$Filesystem::MERGE_DIR/$_");
	    }
	    else {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::INCOMPLETE_DIR") if(-e "$Filesystem::MERGE_DIR/$_");
	    }
	    ###	++$number_missing;
	}
	close BAD;
	
	&$System::stage_step("Move the log file to DARTS staging");
	&$System::perform("$Program::MV $Filesystem::HEADLESS_PACKETS_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/headless_packets.$Dates::start_date_string.txt");
    }
    else {
	&$System::comment("No headless packets file");
    }
}

sub perform_incomplete_check {
    Trace::trace((caller(0))[3]);
    &$System::stage_step("Check which mission data files end with incorrect ccsds packet flags and move them to $Filesystem::NURSERY_DIR");
    if(-e "$Filesystem::INCOMPLETE_PACKETS_LOG") {
	&$System::comment("Incomplete packets file exists");
	open(BAD, "< $Filesystem::INCOMPLETE_PACKETS_LOG");
	while(<BAD>) {
	    next if /^Incomplete/;
	    last if /^$/;
	    chomp;
	    ###	log_msg(*LOG, $state, "Moving $merge_dir/$_ to $merge_dir/incomplete/");
	    if($Flags::RESCUE_MOD) {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::NURSERY_DIR") if(-e "$Filesystem::MERGE_DIR/$_");
	    }
	    else {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::INCOMPLETE_DIR") if(-e "$Filesystem::MERGE_DIR/$_");
	    }
	    ###	++$number_missing;
	}
	close BAD;
	
	&$System::stage_step("Move the log file to DARTS staging");
        &$System::perform("$Program::MV $Filesystem::INCOMPLETE_PACKETS_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/incomplete_packets.$Dates::start_date_string.txt");
    }
    else {
	&$System::comment("No incomplete packets file");
    }
}

sub perform_missing_check {
    Trace::trace((caller(0))[3]);
    
    my $number_missing = 0;
    &$System::stage_step("Check which mission data files have missing packets and move them to $Filesystem::NURSERY_DIR");
    if(-e "$Filesystem::MISSING_PACKETS_LOG") {
	&$System::comment("Missing packet file exists");
	open(BAD, "< $Filesystem::MISSING_PACKETS_LOG");
	while(<BAD>) {
	    next if /^Missing/;
	    last if /^$/;
	    chomp;
	    # TEMP!!!!
	    #	print STDERR "[$_]\n";
	    
	    if($Flags::RESCUE_MOD) {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::NURSERY_DIR");
	    }
	    else {
		&$System::perform("$Program::MV $Filesystem::MERGE_DIR/$_ $Filesystem::INCOMPLETE_DIR");
	    }
	    ++$number_missing;
	}
	close BAD;
	
	&$System::stage_step("Move the log file to DARTS staging");
	&$System::perform("$Program::MV $Filesystem::MISSING_PACKETS_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/missing_packets.$Dates::start_date_string.txt");
	###    log_msg(*LOG, $state, "Number of missing archives: $number_missing");
    }
    else {
	&$System::comment("No missing packet file");
    }
}    

sub perform_non_compressed_check {
    Trace::trace((caller(0))[3]);

#    my @non_compressed_files = ();
    &$System::stage_step("Find out which mission data files are not compressed");
    if(! open(FH, "< $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/md_hdrs.$Dates::start_date_string.html")) {
	&$System::comment("Can't open the md_hdrs file for compression setting check");
    }
    else {
	while(<FH>) {
	    my $filename;
	    next if /^$/;
	    if(/nasA/) {
		chomp;
		$filename = substr($_, -26, 26);
		next;
	    }
	    next if /Type/;
	    next if /headers/;
	    my @stuff = split;
	    my $comp = sprintf("%u", hex($stuff[17]));
	    if(/ccsds/) {
		push(@non_compressed_files, $filename) if $comp == 0;
	    }
	}
	close FH;
    }
}

sub perform_header_dump {

    &$System::stage_step("Perform mission header dump");

}

sub perform_engineering_check {

    &$System::stage_step("Perform engineering check");

}

=head3 Stage functions                                                                                                  
                                                                                                                        
These are functions called from the main pipeline program as major stages in the reformatting process.                  
                                                                                                                        
=cut                                                                                                                    

=pod                                                                                                                    
                                                                                                                        
=over 1                                                                                                                 
                                                                                                                        
=item fetch                                                                                                             
                                                                                                                        
=item Snazzle                                                                                                           
                                                                                                                        
=back                                                                                                                   
                                                                                                                        
=cut                                                                                                                    
# Fetch ccsds packets from SIRIUS
sub fetch {
    Trace::trace((caller(0))[3]);
    my $ant  = 'usc34';	# Not needed for merge as sdtp will substitute merge antenna
    my $band = 3;	# Merge telemetry
    my $mode = 'merge';	# Merge telemetry

#    &$System::stage_title("Data::fetch : Fetch data and place in the received directory ($Filesystem::MERGE_DIR)");

    &$System::stage_step("Fetch ccsds packets");
    # To be able to use receive_dir need to change sdtp.pl
    &$System::perform("$Program::SDTP_FETCH $mode $ant band=$band sdate=$Dates::sdate edate=$Dates::edate stime=$Dates::stime etime=$Dates::etime");

    # Remove status files
    &$System::stage_step("Remove status packets");
    &$System::perform("$Program::RM $Filesystem::MERGE_DIR/eis_sts*");

    # Remove dmp files
    &$System::stage_step("Remove memory dump packets");
    &$System::perform("$Program::RM $Filesystem::MERGE_DIR/eis_dmp*");
}


# Ccsds packet checks
sub check {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::check : Mission data check");

    Data::init_logs();
    
    &$System::stage_step("Count number of received eis_md files");
    @received_files = &$System::list($Filesystem::MERGE_DIR, "eis_md*");

    if(! @received_files) {
	&$System::comment("No ccsds mission data files");
#	# Does the current date already exist in the controller file?
	&$System::comment("Check whether current date in the controller file");
	&$System::comment("\$exists = `$Program::GREP '$Dates::sdate $Dates::edate' $Filesystem::MISSION_PENDING_FILE`;");
	my $exists = `$Program::GREP '$Dates::sdate $Dates::edate' $Filesystem::MISSION_PENDING_FILE`;
	
#	if(Delta_Days($syear1,$smonth1,$sday1,$today[5],$today[4],$today[3]) < 14) {
#	    # Write out start/end date to controlling file if not
#	    `/bin/echo $sdate $edate $stime $etime >> ~/track/mission_pending.txt` if $exists eq "";
#	}
#	else {
#	    print MLOG "\t*** Giving up with plan for $sdate $edate $stime $etime - no ccsds packets ***\n";
#	}

	if (Dates::day_difference() < 14) {
	    &$System::perform("echo -p $Dates::sdate $Dates::edate $Dates::stime $Dates::etime >> $Filesystem::MISSION_PENDING_FILE") if $exists eq "";
	}
	else {
	    Pipeline::main_log_message("\t*** Giving up with plan for $Dates::sdate $Dates::edate $Dates::stime $Dates::etime - no ccsds packets ***\n");
	}
	&$System::perform("$Program::CP $Filesystem::MISSION_PENDING_FILE $Filesystem::DARTS_STAGING_DIR");

	Pipeline::main_log_message("No ccsds packets available");
	#Pipeline::pipeline_exit(1, "No ccsds packets available");
	#Pipeline::pipeline_die(1, "No ccsds packets available");
	Pipeline::pipeline_warn(1, "No ccsds packets available");
    }
    
    &$System::comment("Number of received ccsds files: " . @received_files);
    
    #
    # Write out all files received
    #
# experimental                                                                                                          
    System::dprint("QWERTYUIOP");
#    &$System::perform("System::dprint('FOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO');");
#    &$System::perform("if(open(FH, \">$Filesystem::RECEIVED_FILES_LOG\")) {                                             
#        close FH;                                                                                                       
#        System::dprint(\"Opened $Filesystem::RECEIVED_FILES_LOG: $!\");                                                 
#    }                                                                                                                   
#    else {                                                                                                              
#        System::dprint(\"Can't open $Filesystem::RECEIVED_FILES_LOG: $!\");                                             
#    }                                                                                                                   
#");
    if(open(FH, ">$Filesystem::RECEIVED_FILES_LOG")) {
        close FH;
	System::dprint("Opened $Filesystem::RECEIVED_FILES_LOG: $!");
    }
    else {
	System::dprint("Can't open $Filesystem::RECEIVED_FILES_LOG: $!");
    }

#    my $code = {                                                                                                       
#       if(open(FH, ">$Filesystem::RECEIVED_FILES_LOG")) {                                                              
#           print FH @received_files;                                                                                   
#           close FH;                                                                                                   
#       }                                                                                                               
#       else {                                                                                                          
#           &$System::comment("Can't open $Filesystem::RECEIVED_FILES_LOG: $!");                                        
#       }                                                                                                               
#    };                                                                                                                 
# end of experiment

    &$System::stage_step("Write received filenames to $Filesystem::RECEIVED_FILES_LOG");
    if(open(FH, ">$Filesystem::RECEIVED_FILES_LOG")) {
	print FH @received_files;
	close FH;
    }
    else {
	&$System::comment("Can't open $Filesystem::RECEIVED_FILES_LOG: $!");
    }


####    &$System::run_stage(\&Ccsds::perform_packet_checks, "Ccsds::perform_packet_checks : Perform ccsds packet checks on the received files");                                                                                               
    
#    # Do the ccsds check before joining
#    perform(*LOG, $state, "$HOME/bin/pipeline_md_hdr_dump_pre_join $merge_dir > $merge_dir/md_hdr_dump_pre_join.txt");
#    perform(*LOG, $state, "$HOME/bin/pipeline_ccsds_pre_split_check.pl $merge_dir > $merge_dir/ccsds_hdr_pre_join.txt");

    ###!!!
    &$System::stage_step("Perform mission data split check", "$Program::MD_SPLIT_CHECK $Filesystem::MERGE_DIR > $Filesystem::MD_SPLIT_CHECK_LOG");

    &$System::stage_step("Perform mission data split check");
    &$System::perform("$Program::MD_SPLIT_CHECK $Filesystem::MERGE_DIR > $Filesystem::MD_SPLIT_CHECK_LOG");
    ###!!!

    &$System::stage_step("Perform mission data join");
    &$System::perform("$Program::MD_JOIN $Filesystem::MD_SPLIT_CHECK_LOG");

    # mcrw 20130902 - put in absolute path temporarily
    #print "x","$HOME","x\n";

    &$System::stage_step("Count number of joined files");
    @joined_files = &$System::list($Filesystem::JOIN_DIR, "eis_md*");
    
#    $log_msg_str = "Number of joined files: " . @joined_files;
#    &$System::comment($log_msg_str);
    &$System::comment("Number of joined files: " . @joined_files);
    
    #
    # Write out all the joined files
    #
    &$System::stage_step("Write joined filenames to $Filesystem::JOINED_FILES_LOG");
### dmm1 if(open(FOO, ">$merge_dir/joined_files.txt")) {
    if(open(FH, ">$Filesystem::JOINED_FILES_LOG")) {
	print FH @joined_files;
	close FH;
    }
    else {
	&$System::comment("Can't open $Filesystem::JOINED_FILES_LOG: $!");
    }
    
    if(!@joined_files) {
	&$System::comment("No files in the join directory - quitting");
	Pipeline::pipeline_exit(-1000, "No files in the join directory");
    }
    
    #
    # Remove the files currently in the merge directory with the joined ones
    #
    &$System::stage_step("Replace the received files with the joined ones");
    &$System::perform("$Program::RM $Filesystem::MERGE_DIR/eis_md_*");

#    # mcrw 20130904 - changed path
    &$System::perform("$Program::MV $Filesystem::JOIN_DIR/eis_md* $Filesystem::MERGE_DIR");
#    &$System::perform("/bin/mv /nasA_solar1/home/sbukeis/tmp/join/* $Pipeline::MERGE_DIR");
    
    # Do the mission data header dump, which also updates the engineering.txt file
    &$System::stage_step("Perform mission header dump");
    Ccsds::ccsds_md_packet_dump($System::JOIN_DIR, $System::MD_HEADER_DUMP_FILE);

    # Or Ccsds:: ?                                                                                                
    #Data::perform_header_dump();                                                                                 
    #Data::perform_engineering_check();

    # Do the ccsds check which detects missing packets, updating missing_packets.txt
    &$System::stage_step("Perform the ccsds packet check");
    #&$System::perform("$Program::CCSDS_PACKET_CHECK $Filesystem::MERGE_DIR > $Filesystem::CCSDS_PACKET_CHECK_LOG");
    Ccsds::ccsds_packet_dump($System::JOIN_DIR, $Filesystem::CCSDS_PACKET_DUMP_FILE);

    ###goto end_join_test;

    # Move the files to DARTS
    &$System::stage_step("Move check logs to DARTS staging");
########    &$System::perform("/bin/mv $Filesystem::MD_HEADER_CHECK_LOG $Filesystem::DARTS_MISSION/$Dates::year/$Dates::month/packet_check/md_hdrs.$Dates::start_date_string.html");
#    &$System::perform("/bin/mv $Filesystem::MD_HEADER_DUMP_FILE $darts_md_check_log");
    &$System::perform("$Program::MV $Filesystem::MD_HEADER_DUMP_FILE $Filesystem::DARTS_MD_CHECK_LOG");

    &$System::perform("$Program::MV $Filesystem::CCSDS_PACKET_CHECK_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/ccsds_hdrs.$Dates::start_date_string.txt");
# perform(*LOG, $state, "/bin/mv $merge_dir/ccsds_hdr_pre_join.txt   $darts_mission/$year/$month/packet_check/ccsds_hdrs_pre_join.$year$month$day.txt");
   
    &$System::perform("$Program::MV $Filesystem::LOCAL_SHUTTER_LOG $Filesystem::DARTS_SHUTTER_LOG_DIR/shutter_$Dates::start_date_string.txt");
    &$System::perform("$Program::MV $Filesystem::RECEIVED_FILES_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/received.$Dates::start_date_string.txt");
    &$System::perform("$Program::MV $Filesystem::MD_SPLIT_CHECK_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/md_split_check.$Dates::start_date_string.txt");
    &$System::perform("$Program::MV $Filesystem::JOINED_FILES_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/joined_files.$Dates::start_date_string.txt");
    
    #
    # Check which files are not compressed. Check to safeguard correct operation of the new rescue c code
    # Do for next version which does not constantly move files about the place
    #perform_non_compressed_check();
    
    #
    # For now, if there are any archives with missing packets move them to the incomplete directory.
    # Must handle these more sensibly in the future.
    #
    Data::perform_missing_check();
    
    #
    # Check for and record those files which are continuations of previous files.
    # The decompression stage will move these files aside.
    #
    Data::perform_headless_check();
    
    #
    # Check for and record those files which are not complete.
    # The decompression stage will move these files aside.
    #
    Data::perform_incomplete_check();
    
    &$System::stage_step("Move the sequence counts log file to DARTS staging");
    &$System::perform("$Program::MV $Filesystem::SEQUENCE_COUNTS_LOG $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/sequence_counts_$Dates::start_date_string.txt");
    
  end_join_test:
    
    # Get a count of number of mission data files here
    #@pre_decompression = `ls $merge_dir/eis_md*`;
    # mcrw 20130802
    &$System::stage_step("Count the number of undamaged files prior to decompression");
    my @pre_decompression = &$System::list($Filesystem::JOIN_DIR, "eis_md*");
    
    $number_pre_decompression = @pre_decompression;
    &$System::comment("Number of compressed files: " . scalar(@pre_decompression));
    &$System::comment("Number of compressed files: $number_pre_decompression");
    
    Pipeline::pipeline_exit(0, "FORCE_REFORMAT") if $Flags::FORCE_REFORMAT;
    Pipeline::pipeline_exit($Flags::codes{FORCE_REFORMAT}[0], $Flags::codes{FORCE_REFORMAT}[1]) if $Flags::FORCE_REFORMAT;
}

sub decompress_alt {
    Trace::trace((caller(0))[3]);
    &$System::stage_title("Data::decompress_alt : Decompress mission data (alternative version)");

    # All the compressed files are left in the joined location

    &$System::stage_step("Run the jpeg recover program over each file");

    &$System::code_log("ls $Filesystem::JOIN_DIR");
    my @compressed_files = `ls $Filesystem::JOIN_DIR`;
    foreach my $file (@compressed_files) {
	chomp $file;
	&$System::perform("$Program::RESCUE $Filesystem::JOIN_DIR/$file $Filesystem::MERGE_DIR/$file");
    }

    &$System::stage_step("Prepare the IDL decompression script");
    &$System::comment("(IDL) eis_md_decomp_chk_tbl_fix1_script,'$Filesystem::MERGE_DIR', /merge\n");

    my $output = "!quiet=1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_ccsds_interface__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/mdppmdctrecover\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_chk_tbl_fix1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_chk_tbl_fix1_script\n";
    $output .= "eis_md_decomp_chk_tbl_fix1_script,'$Filesystem::MERGE_DIR', /merge\n";
    
    open(FH, "> $Program::DECOMP_ALT");
    print FH $output;
    close FH;
    
    &$System::comment("IDL file:\n$output");

    # Start the mission data decompress
    &$System::stage_step("Start the mission data decompress");
    my $decompression_log = $Filesystem::LOCAL_DECOMPRESSION_LOG . "$Dates::start_date_string" . ".txt";
    &$System::perform("$Program::SSWIDL < $Program::DECOMP_ALT >& $decompression_log");

    # Copy the decompression record and logs to DARTS
    &$System::stage_step("Copy the decompression logs to DARTS");
    &$System::perform("$Program::CP $Filesystem::DECOMPRESSION_RECORD_LOG $Filesystem::DARTS_STAGING_LOGS/decompression");

    my $decompress_log = $Filesystem::DARTS_STAGING_LOGS . "/decompression/md_decompress_" . "$Dates::start_date_string" . ".txt";
    &$System::perform("$Program::SLIM < $decompression_log > $decompress_log");

    my $alt_decompress_log = $Filesystem::DARTS_STAGING_LOGS . "/decompression/md_decompress_" . "$Dates::start_date_string" . "_alt.txt";
    &$System::perform("/bin/cat $decompress_log | awk -f $Program::ALT_SLIM > $alt_decompress_log");

    # mcrw 20190808 - temp, just to get fuller log
    &$System::perform("$Program::CP $decompression_log $Filesystem::DARTS_STAGING_LOGS/decompression");

    # Get a count of number of mission data files here
    # mcrw 20130802
    &$System::stage_step("Get a count of the decompressed files");
####    my @post_decompression = `/bin/ls $System::DECOMPRESSED_DIR | /usr/bin/grep eis_md`;
    my @post_decompression = &$System::list($Filesystem::DECOMPRESSED_DIR, "eis_md*");

    $number_post_decompression = @post_decompression;
    &$System::comment("Number of de-compressed files: " . scalar(@post_decompression));

}

# Kick off the decompression
sub decompress {
    Trace::trace((caller(0))[3]);
#    &$System::stage_title("Data::decompress : Decompress mission data");

    &$System::stage_step("Prepare the IDL decompression script");
    &$System::comment("(IDL) eis_md_decomp_fix1_script,'$Filesystem::MERGE_DIR', /merge");
    create_decompress_idl_program();
    
    # Start the mission data decompress
    &$System::stage_step("Start the mission data decompress");
    my $decompression_log = $Filesystem::LOCAL_DECOMPRESSION_LOG . "$Dates::start_date_string" . ".txt";
    &$System::perform("$Program::SSWIDL < $Program::MD_DECOMP >& $decompression_log");

    # Copy the decompression record and logs to DARTS
    &$System::stage_step("Copy the decompression record to DARTS");
    &$System::perform("$Program::CP $Filesystem::DECOMPRESSION_RECORD_LOG $Filesystem::DARTS_STAGING_LOGS/decompression");

    #??
    #log_msg(*LOG, $state, "$Program::MV $pipeline_log/md_decompress_$date_string.txt $Pipeline::DARTS_STAGING_LOGS/decompression");
    #system("$Program::MV $pipeline_log/md_decompress_$date_string.txt $Pipeline::DARTS_STAGING_LOGS/decompression");

    &$System::stage_step("Reduce the decompression log output");
    my $decompress_log = $Filesystem::DARTS_STAGING_LOGS . "/decompression/md_decompress_" . "$Dates::start_date_string" . ".txt";
    &$System::perform("$Program::SLIM < $decompression_log > $decompress_log");

    my $alt_decompress_log = $Filesystem::DARTS_STAGING_LOGS . "/decompression/md_decompress_" . "$Dates::start_date_string" . "_alt.txt";
    &$System::perform("$Program::CAT $decompress_log | awk -f '$Program::ALT_SLIM' > $alt_decompress_log");

    # mcrw 20190808 - temp, just to get fuller log
    &$System::stage_step("Copy the decompression log to DARTS");
    &$System::perform("$Program::CP $decompression_log $Filesystem::DARTS_STAGING_LOGS/decompression");

    # mcrw 20190130 (20190806 - comment out)
    # Remove the decompress_log too - just use the alt version from now
    #perform(*LOG, $state, "$Program::RM $decompress_log");

    ### REMOVE md_decompress_$date_$string file here

    # Get a count of number of mission data files here
    # mcrw 20130802
    #@post_decompression = `ls $merge_dir/decompressed/ | grep eis_md`;
    &$System::stage_step("Get a count of the decompressed files");
    my @post_decompression = &$System::list($Filesystem::DECOMPRESSED_DIR, "eis_md*");

    $number_post_decompression = @post_decompression;
    &$System::comment("Number of de-compressed files: " . scalar(@post_decompression));
    
#    if($number_post_decompression != 0) {
#	&$System::perform("$Program::MD_HEADER_DUMP $Pipeline::DECOMPRESSED_DIR > $Pipeline::MD_HEADER_CHECK_LOG");
#	&$System::perform("$Program::CCSDS_PACKET_CHECK $Pipeline::DECOMPRESSED_DIR > $Pipeline::CCSDS_PACKET_CHECK_LOG");
#	&$System::perform("$Program::MV $Pipeline::MD_HEADER_CHECK_LOG $System::DARTS_MISSION/$Dates::year/$Dates::month/packet_check/decompressed_md_hdrs.$Dates::start_date_string.html");
#	&$System::perform("$Program::MV $$Pipeline::CCSDS_PACKET_CHECK_LOG $System::DARTS_MISSION/$Dates::year/$Dates::month/packet_check/decompressed_ccsds_hdrs.$Dates::start_date_string.txt");
#	&$System::perform("gzip -f $System::DARTS_MISSION/$Dates::year/$Dates::month/packet_check/decompressed_md_hdrs.$Dates::start_date_string.html");
#	&$System::perform("gzip -f $System::DARTS_MISSION/$Dates::year/$Dates::month/packet_check/decompressed_ccsds_hdrs.$Dates::start_date_string.txt");
#    }
    
}

# Kick off the FITS reformat process
sub reformat {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::reformat : Mission data reformat");

    &$System::comment("(IDL) fits_script, '$Filesystem::DECOMPRESSED_DIR', '$Filesystem::TEMP_FITS_DIR'");

    #??
    #perform(*LOG, $state, "/bin/chmod 644 /nasA_solar1/home/sbukeis/eisco/planning_db/timeline_db/eis_science_db.dbf");

#####    $translate_log_name = "md_translation" . "_$date_string" . ".txt";
    my $reformat_log = $Filesystem::LOCAL_MD_TRANSLATION_LOG . "$Dates::start_date_string" . ".txt";
    $Date::translate_log = $Filesystem::LOCAL_LOG_DIR . "/md_translation_" . $Dates::start_date_string . ".txt";

    # mcrw 20190809 added code to able older studies be re-reformatted (avoid sequence over-flow bug)
    my $old_seq_str = ""; #($OLD_SEQ_FLAG == 1) ? ", /noalter_seq" : "";
#    $reformat_log = $Filesystem::LOCAL_MD_REFORMAT_LOG . "$Dates::start_date_string" . ".txt";

    create_reformat_idl_program($reformat_log, $old_seq_str);

    #$reformat_log = "/dev/null";
    
    # Changed cp to mv
#    &$System::comment("Run $MissionProgram::MD_REFORMAT");
    &$System::stage_step("Run $Program::MD_REFORMAT");
    &$System::perform("$Program::SSWIDL < $Program::MD_REFORMAT >& $reformat_log");
    
    &$System::stage_step("Copy reformat logs to DARTS");
    &$System::perform("$Program::CP $reformat_log $Filesystem::DARTS_STAGING_DIR/logs/md_reformat/$Dates::year");
    &$System::perform("$Program::CP $Date::translate_log $Filesystem::DARTS_STAGING_DIR/logs/md_translation/$Dates::year");
    
    # Get a count of number of mission data files here
    &$System::stage_step("Get count of number of fits files");
    my @post_reformat = &$System::list($Filesystem::FITS_DIR, "eis_l0*");

    &$System::comment("Number of reformatted files: " . scalar(@post_reformat));
    
    # TODO: Compare the number of output files with the number of decompressed files. If not the same
    # then restart the reformat after separating out the done from the pending. Assume the first pending
    # file caused the reformat to crash and move that file to a nursery somewhere.
    
}

# rescue_decomp_chk_tbl.pl
sub rescue_decompress {
    Trace::trace((caller(0))[3]);
    &$System::stage_title("Data::rescue_decompress : Decompress rescued mission data");

    &$System::stage_step("Remove empty files");
####    my @files = `ls $System::NURSERY_RESCUED_DIR/eis_md*`;
    my @files = &$System::list($Filesystem::NURSERY_RESCUED_DIR, "eis_md*");
    my $pre_check_count = @files;
    &$System::comment("Files for rescue:\n@files\n");
    my $zero_sized_count = 0;
    foreach my $f (@files) {
	chomp $f;
	my @stat = stat $f;
	if($stat[7] == 0) {
	    ++$zero_sized_count;
	    my $f1 = basename($f);
	    push @empty_files, $f1;
	    &$System::comment("$f is zero-sized, removing\n");
	    &$System::perform("$Program::RM $Filesystem::NURSERY_RESCUED_DIR/$f1");
	}
    }
    my $post_check_count = $pre_check_count - $zero_sized_count;
    my $post_check_count_str = ($post_check_count == 1) ? "file" : "files";
    &$System::comment("$pre_check_count $post_check_count_str sent for rescue, $zero_sized_count empty, $post_check_count $post_check_count_str rescued");

    &$System::stage_step("Count number of files for decompression");
    my @rescued_files = `ls $Filesystem::NURSERY_RESCUED_DIR/eis_md*`;
    &$System::comment("Files for rescue: @rescued_files\n");

    &$System::comment("(IDL) eis_md_decomp_chk_tbl_fix1script,'$Filesystem::NURSERY_RESCUED_DIR', /merge");

    my $output = "!quiet=1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_ccsds_interface__define\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/eis_md_decomp_script\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/mdppmdctrecover\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_chk_tbl_fix1\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_md_decomp_chk_tbl_fix1_script\n";
    $output .= "eis_md_decomp_chk_tbl_fix1_script,'$Filesystem::NURSERY_RESCUED_DIR', /merge\n";

    &$System::stage_step("Write to $Program::RESCUE_DECOMP");
    open(MD_DECOMP, "> $Program::RESCUE_DECOMP");
    print MD_DECOMP $output;
    close MD_DECOMP;

# Start the mission data decompress

    &$System::stage_step("Perform decompression");
    my $output_log = "$Filesystem::LOCAL_DECOMPRESSION_LOG" . "rescue_" . $Dates::start_date_string . ".txt";
    &$System::perform("$Program::SSWIDL < $Program::RESCUE_DECOMP >& $output_log");
###    &$System::log("$Program::CP $System::DECOMPRESSION_RECORD_LOG $System::DARTS_STAGING_LOGS/decompression/");

    &$System::comment("$Program::CP $Filesystem::DECOMPRESSION_RECORD_LOG $Filesystem::DARTS_STAGING_LOGS/decompression/");

    my $decompress_log = "$Filesystem::DARTS_STAGING_LOGS" . "/decompression/rescue/md_decompress_rescue_" . "$Dates::start_date_string" . ".txt";
    my $alt_decompress_log = "$Filesystem::DARTS_STAGING_LOGS" . "/decompression/rescue/md_decompress_rescue_" . "$Dates::start_date_string" . "_alt.txt";

    &$System::stage_step("Slim decompression log");
    &$System::perform("$Program::SLIM < $output_log > $decompress_log");
    &$System::perform("$Program::CAT $decompress_log | awk -f '$Program::SLIM_RESCUE' > $alt_decompress_log");

# mcrw 20190130
# Remove rescue decompress log too
    &$System::stage_step("Remove decompression logs");
    &$System::perform("$Program::RM $decompress_log");
    &$System::perform("$Program::RM  $output_log");

    &$System::stage_step("Count number of decompressed files");
####    my @post_decompression = `/bin/ls $System::RESCUE_DECOMPRESSED_DIR | /bin/grep eis_md`;
    my @post_decompression = &$System::list($Filesystem::RESCUE_DECOMPRESSED_DIR, "eis_md*");
    my $number_post_decompression = @post_decompression;
    &$System::comment("Number of de-compressed files: $number_post_decompression");
    &$System::comment("rescue_decomp finished");
}

# rescue_reformat_chk_tbl.pl
sub rescue_reformat {
    Trace::trace((caller(0))[3]);
    &$System::stage_title("Data::rescue_reformat : Starting rescued mission data reformat");

    my $translate_log_name = "md_reformat_rescue" . "_" . "$Dates::start_date_string" . ".txt";
    my $translate_log = "$Filesystem::LOCALDATA_LOG_DIR" . "/$translate_log_name";

    my $output = "!quiet=0\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/merge/db_read_sciencestudyraster_entry\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/eis_read_orbit_events_reformatter\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/merge/eis_hdr__define\n";
    $output .= ";.comp $Filesystem::LOCAL_IDL/merge/eis_data__readfits\n";

# mcrw 20190806 - uncommented next line to make it the same as in daily_merge_mission1
# for the undamaged files
    $output .= ";.comp $Filesystem::LOCAL_IDL/merge/eis_plan__define\n";

    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_cal__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_data__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_data__readccsds\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_fits_calib\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_fits_coord\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_fits_dataid\n";

    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_spcd__define\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_fits_script\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/merge/eis_mkfits\n";
    $output .= "eis_fits_script,'$Filesystem::DECOMPRESSED_DIR', '$Filesystem::RESCUE_FITS_DIR',logfile='$translate_log', /rescued\n";
    $output .= ";;;eis_fits_script,'$Filesystem::DECOMPRESSED_DIR', '$Filesystem::RESCUE_FITS_DIR',logfile='$translate_log', /nospcd\n";

    &$System::stage_step("Write to IDL file");
    open(MD_FITS, "> $Program::MD_FITS");	#!!!!!!!!!!!!!
    print MD_FITS $output;
    close MD_FITS;

    &$System::comment("IDL file:\n$output");

    my @post_decompression = `ls $Filesystem::DECOMPRESSED_DIR | grep eis_md`;
    my $number_post_decompression = @post_decompression;

    my $total_reformat_count = 0;
    my $more_files = 1;
    my $reformat_log;

    &$System::stage_step("Perform reformat");
    while($more_files) {
	$reformat_log = "$Filesystem::LOCALDATA_LOG_DIR/md_reformat_rescue" . "_$Dates::start_date_string" . ".txt";
	&$System::perform("$Program::SSWIDL < $Program::MD_FITS >> $reformat_log");

	last; # debug

	# Get a count of number of mission data files here
	my @reformat_count = &$System::list($Filesystem::RESCUE_FITS_DIR, "eis_l0*");
	my $reformat_count = @reformat_count;
	my $total_reformat_count += $reformat_count;
	last if $total_reformat_count == $number_post_decompression;
    
	# Reformatter failed on a file.
	# Remove decompressed files which have been reformatted.
	# Move existing fits to temp_fits directory
	# Move failed source file back to incomplete directory.
	# Start reformat again
	if($reformat_count > 0) {
	    while($reformat_count--) {
		my $file = shift @post_decompression;
		chomp $file;
		&$System::perform("$Program::RM $Filesystem::DECOMPRESSED_DIR/$file");
	    }
	}
	my $file = shift @post_decompression;
	chomp $file;
	&$System::perform("$Program::MV $Filesystem::NURSERY_DIR/$file $Filesystem::INCOMPLETE_DIR");
	&$System::perform("$Program::RM $Filesystem::DECOMPRESSED_DIR/$file");
	&$System::perform("$Program::MV $Filesystem::RESCUE_FITS_DIR/* $Filesystem::TEMP_FITS_DIR");
	&$System::perform("echo >> $reformat_log");
	&$System::perform("echo $file failed. Restarting. >> $reformat_log");
	&$System::perform("echo >> $reformat_log");
	++$total_reformat_count;
	$more_files = ($total_reformat_count < $number_post_decompression);
    }

    &$System::stage_step("Copy reformat logs to DARTS");
    &$System::perform("$Program::MV $Filesystem::RESCUE_FITS_DIR/* $Filesystem::TEMP_FITS_DIR");
    &$System::perform("$Program::CP $reformat_log $Filesystem::DATA_STAGING_LOGS_DIR/md_reformat/rescue/");
    &$System::perform("$Program::CP $translate_log $Filesystem::DATA_STAGING_LOGS_DIR/md_translation/rescue/");

# Get a count of number of mission data files here
    &$System::stage_step("Count number of reformatted files");
####    my @post_reformat = `/bin/ls $System::TEMP_FITS_DIR/eis_l0*`;
    my @post_reformat = &$System::list($Filesystem::TEMP_FITS_DIR, "eis_l0*");
    &$System::comment("Number of reformatted files: " . scalar(@post_reformat));

}

# Rescue damaged files
sub rescue {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::rescue : Rescue damaged files");

    if(@non_compressed_files) {
    # If there are non compressed files then send them to the old rescue method
	# perform(*LOG, $state, "$Program::MV $merge_dir/decompressed/eis_md* $merge_dir/merge1/decompressed1/");
    }

    &$System::stage_step("Copy the damaged files to the rescue directory?") if $Flags::RESCUE_MOD;
    &$System::perform("$Program::CP $Filesystem::NURSERY_DIR/eis_md* $Filesystem::RESCUE_DIR/") if $Flags::RESCUE_MOD;
    
    if(($Flags::USE_NEW_RESCUE) || ($Flags::SPECIAL eq "special")) {
	#&$System::perform($Program::CHECK_TABLE_RESCUE . $Dates::start_date_string . "_" . $Dates::eday);
	&$System::comment("Using jpeg rescue");
	jpeg_rescue();
    }
    else {
	#&$System::perform($Program::RESCUE . $Date::sdate . "_" . $Dates::eday);
	&$System::comment("Using old rescue");
	old_rescue();
    }
    
    rescue_decompress();

    rescue_reformat();

    # Perform ccsds packet check to see how the rescue performed.
    #@rescued_files = `ls $merge_dir/ | grep eis_md`;
    #if(! @rescued_files) {
    #    log_msg(*LOG, $state, "No rescued ccsds mission data files");
    #}
    #else {
    #
    #    log_msg(*LOG, $state, "Perform packet check on rescued ccsds mission data files");
    #    perform(*LOG, $state, "$HOME/bin/pipeline_ccsds_check $merge_dir > $merge_dir/ccsds_hdr_check.txt");
    #    perform(*LOG, $state, "$Program::MV $merge_dir/ccsds_hdr_check.txt $darts_mission/$year/$month/packet_check/rescued_ccsds_hdrs.$year$month$day.txt");
    #
    #}
    
    
    # Copy decompressed files from decompressed1 to decompressed
    #perform(*LOG, $state, "$Program::CP $merge_dir/merge1/decompressed1/eis_md* $merge_dir/decompressed/");
    
    # Get count of decompressed files again
    #@total_decompressed_files = `ls $merge_dir/decompressed/eis_md*`;
    
    # add log files to this program's log 
    
    # Get the number of fits files after rescue
    &$System::stage_step("Count number of files after rescue reformat");
    my @pre_decompression = &$System::list($Filesystem::NURSERY_DIR, "eis_md*");
    $number_pre_decompression += scalar(@pre_decompression);
    my @post_decompression = &$System::list($Filesystem::RESCUE_DECOMPRESSED_DIR, "eis_md*");
    $number_post_decompression += scalar(@post_decompression);
    my @post_reformat = &$System::list($Filesystem::RESCUE_FITS_DIR, "eis_l0*");
    
    
    #if($number_post_decompression != 0) {
    #    perform(*LOG, $state, "$HOME/bin/pipeline_md_hdr_dump $merge_dir/decompressed > $merge_dir/md_hdr_check.txt");
    #    perform(*LOG, $state, "$HOME/bin/pipeline_ccsds_check $merge_dir/decompressed > $merge_dir/ccsds_hdr_check.txt");
    #
    #    perform(*LOG, $state, "$Program::MV $merge_dir/md_hdr_check.txt $darts_mission/$year/$month/packet_check/rescued_decompressed_md_hdrs.$year$month$day.html");
    #    perform(*LOG, $state, "$Program::MV $merge_dir/ccsds_hdr_check.txt $darts_mission/$year/$month/packet_check/rescued_decompressed_ccsds_hdrs.$year$month$day.txt");
    
    
    &$System::stage_step("Gzip header/packet dump files");
    &$System::perform("$Program::GZIP -f $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/rescued_decompressed_md_hdrs.$Dates::start_date_string.html");
    &$System::perform("$Program::GZIP -f $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/rescued_decompressed_ccsds_hdrs.$Dates::start_date_string.txt");
    
    
    #
    # Write out all the fits file names
    #
    &$System::stage_step("Print fits filenames to $Filesystem::MERGE_DIR/fits_files.txt");
    if(open(FH, ">$Filesystem::MERGE_DIR/fits_files.txt")) {
	print FH @post_reformat;
	close FH;
    }
    else {
	&$System::comment("Can't open $Filesystem::MERGE_DIR/fits_files.txt: $!");
    }
    &$System::stage_step("Move fits files log to DARTS");
    &$System::perform("$Program::MV $Filesystem::MERGE_DIR/fits_files.txt $Filesystem::DARTS_DIR/$Dates::year/$Dates::month/packet_check/fits_files.$Dates::start_date_string.txt");
}

# Update the raw to FITS translations
sub update_translation_databases {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::update_translation_databases : Update md translation database");

####    &$System::comment("Deprecated.");
####    return;

# eis_mkfits writes all the ccsds packet filenames and the corresponding fits filenames processed
# to a text file. Open the text file and transfer the information to a perl database.

    # temporary
    my $pipeline_log = "PIPELINE_LOG";
    my $translate_log = "TRANSLATE_LOG";
    my %temp;
    my $darts_staging = "DARTS_STAGING";
    my $sdate = "SDATE";
    my $eday = "EDAY";

    # This has changed remotely...
    my $db_open = 0;
    if(dbmopen(my %trans, "$pipeline_log/mission_name_dbase", 0644)) {
	$db_open = 1;
	&$System::comment("$pipeline_log/mission_name_dbase opened ok");
	if(open(REF_LOG, "< $translate_log")) {
	    &$System::comment("Opened $translate_log ok");
	    while(<REF_LOG>) {
                my @files = split;         # Split ccsds filename and fits filename
                my $ccsds = $files[0];
                my $fits = $files[1];
                $ccsds =~ s/ //g;               # Remove any white space from names
                $fits =~ s/ //g;
                chomp $fits;            # Remove newline if any
                $trans{$ccsds} = $fits;
                &$System::comment("Updating mission_name_database: $ccsds => $fits");
                $temp{$ccsds} = $fits;
	    }
	    close(REF_LOG);
	    &$System::comment("Closed $translate_log");
	}
	else {
	    &$System::comment("Can't open $translate_log: $!");
	}
	
        my $rescue_translation_log = "$darts_staging/logs/md_translation/rescue/md_translation_rescue_" . "$sdate" . "_" . "$eday" . ".txt";
	if(open(REF_LOG, "< $rescue_translation_log")) {
	    &$System::comment("Opened $translate_log ok");
	    while(<REF_LOG>) {
                my @files = split;         # Split ccsds filename and fits filename
                my $ccsds = $files[0];
                my $fits = $files[1];
                $ccsds =~ s/ //g;               # Remove any white space from names
                $fits =~ s/ //g;
                chomp $fits;            # Remove newline if any
                $trans{$ccsds} = $fits;
                &$System::comment("Updating mission_name_database: $ccsds => $fits");
                $temp{$ccsds} = $fits;
	    }
	    close(REF_LOG);
	    &$System::comment("Closed $translate_log");
	}
	else {
	    &$System::comment("Can't open $rescue_translation_log: $!");
	}
	
	&$System::comment("Closing mission_name_dbase");
	if(dbmclose(%trans)) {
	    &$System::comment("Closed mission_name_dbase ok");
	}
	else {
	    &$System::comment("Closing mission_name_dbase failed: $!");
	}
    }
    else {
	&$System::comment("Cannot open mission_name_dbase: $!");
    }
    
# Copy the database to the staging area even if it was not updated
    &$System::perform("$Program::CP $pipeline_log/mission_name_dbase $darts_staging/logs/");
    
# Remove translate_log, reformat_log?
    
}

# Moved to Fits.pm
=pod
# Generate the FITS header dump
sub fits_header_dump {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::fits_header_dump : Generating fits header dump");

    my $fits_dump_log = $Filesystem::LOCAL_FITS_DUMP_LOG . $Dates::start_date_string . ".txt";
    my $fits_headers_log = $Filesystem::LOCAL_FITS_HEADERS_LOG . $Dates::start_date_string . ".txt";
    my $fits_headers_log_gzip = $fits_headers_log . ".gz";
    
    &$System::stage_step("Do fits dump");
    &$System::perform("$Program::FITS_STUDY_METRIC $Filesystem::TEMP_FITS_DIR/* > $fits_dump_log");

    # mcrw 20130918 - needs fixing
    &$System::stage_step("Do fits headers dump");
    &$System::perform("$Program::FITS_HEADER_DUMP $Filesystem::TEMP_FITS_DIR/* >> $fits_headers_log");

    # Changed cp to mv
    &$System::stage_step("Move logs to DARTS and gzip");
    &$System::perform("$Program::MV $fits_dump_log " . $Filesystem::DARTS_FITS_DUMP_LOG . "/$Dates::year");
    &$System::perform("gzip -f $fits_headers_log");
#???    &$System::perform("$Program::MV $fits_headers_log_gzip " . $System::DARTS_STAGING_FITS_HEADERS . "/$Dates::year");

}
=cut

# Moved to Fits.pm
=pod
# Perform the data loss checks
sub blank_pixel_check {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::blank_pixel_check : Generating blank pixel data");

    my $data_loss_log = $Filesystem::DARTS_STAGING_LOGS . "/$Dates::start_date_string" . "_data_loss.txt";

    # As the IDL appends to the log file remove it before running the idl
    &$System::stage_step("Remove data loss log as IDL appends to it");
    &$System::perform("if [ -e $data_loss_log ] ; then $Program::RM $data_loss_log; fi");

    &$System::comment("(IDL) data_loss, '$Filesystem::TEMP_FITS_DIR',log_file='$data_loss_log'");

    my $output = ".comp $Filesystem::LOCAL_IDL/data_loss\n";
    $output .= "data_loss,'$Filesystem::TEMP_FITS_DIR',log_file='$data_loss_log'\n";

    &$System::stage_step("Create IDL file");
    open(DL, "> $Program::DATA_LOSS");
    print DL $output;
    close DL;

    &$System::comment("IDL file:\n$output");
    &$System::stage_step("Run IDL");
    &$System::perform("$Program::SSWIDL < $Program::DATA_LOSS");
    &$System::stage_step("Copy log to DARTS");
    
    &$System::perform("$Program::RM $Program::DATA_LOSS");

}
=cut

# Moved to Fits.pm
=pod
# Dump the check tables to DARTS
sub check_table_dump {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::check_table_dump : Generating check table dump");

    my $chk_table_dump_file = $Filesystem::DARTS_STAGING_CHECK_TABLE_DUMP . $Dates::sdate . "_" . $Dates::eday . ".txt";
    my $chk_table_dump_store = $Filesystem::DARTS_STAGING_CHECK_TABLE_STORE . $Dates::sdate . "_" . $Dates::eday;
    
    &$System::comment("(IDL) dump_all_chk_files,log_file='$chk_table_dump_file'");
    
    my $output = ".comp $Filesystem::LOCAL_IDL/chk_file_dump\n";
    $output .= ".comp $Filesystem::LOCAL_IDL/dump_all_chk_files\n";
    $output .= "dump_all_chk_files,LOG_FILE='$chk_table_dump_file'\n";

    &$System::stage_step("Write to IDL file");
    open(IDL, "> $Program::CHECK_DUMP");
    print IDL $output;
    close IDL;
    
    &$System::comment("IDL file:\n$output");
    &$System::stage_step("Run IDL");
    &$System::perform("$Program::SSWIDL < $Program::CHECK_DUMP");
    &$System::stage_step("Remove IDL file");
    &$System::perform("$Program::RM $Program::CHECK_DUMP");
    &$System::stage_step("Gzip table files");
    &$System::perform("gzip -f $Filesystem::CHECK_TABLE_DIR/eis_md*"); #???!!! FIXME
    &$System::stage_step("Move table files to DARTS");
    &$System::perform("/bin/mkdir $chk_table_dump_store");
    &$System::perform("$Program::MV $Filesystem::CHECK_TABLE_DIR/eis_md* $chk_table_dump_store");
    &$System::perform("gzip -f $chk_table_dump_store/*");
    
}
=cut

# Moved to Fits.pm
=pod
# Compress the FITS files
sub compress_fits {
    Trace::trace((caller(0))[3]);

#    &$System::stage_title("Data::compress_fits : Gzip fits files");

    &$System::stage_step("Gzip");
    &$System::perform("/bin/cd $Filesystem::TEMP_FITS_DIR && gzip -f *");

}
=cut


1;

__END__
