#!/usr/bin/env perl

# M. Whillock, m.whillock@ucl.ac.uk, MSSL, UCL, 2006-2023

use strict;
use warnings;

use Trace::Trace;

use Hinode::EIS::EisPipeline::System;
use Hinode::EIS::EisPipeline::Pipeline;
use Hinode::EIS::EisPipeline::Dates;
use Hinode::EIS::EisPipeline::Flags;
use Hinode::EIS::EisPipeline::Args;
use Hinode::EIS::EisPipeline::Data;
use Hinode::EIS::EisPipeline::EisMission::Mission;
use Hinode::EIS::EisPipeline::EisMission::Hk;
use Hinode::EIS::EisPipeline::EisMission::Data;
use Hinode::EIS::EisPipeline::EisMission::Flags;


Trace::trace_on();
Trace::save_to_file(Filesystem::home() . "/tmp/mission_trace.txt");

System::set_timezone_string("(JST)");

# Start the pipeline and open the main pipeline log
Pipeline::create("Mission");

#
# Flags normally reset (0) as default and not controlled by switches:
#   Flags::FORCE_REFORMAT
#   Flags::SKIP_HK
#
# Flags normally set (1) as default and not controlled by switches:
#   Flags::RESCUE_MOD
#   Flags::USE_NEW_RESCUE
#
# Optional command line switches and the flags they set to 1 (reset (0) by default) are:
# --c			Flags::cron				Started by cron
# --cron		Flags::cron				Started by cron
# --pending             Flags::cron, Flags::cron_pending	Started by cron from pending file
# --TEST		Flags::TESTING				General purpose test
# --just_fits		Flags::JUST_FITS_FLAG			Quit after generating the fits files
# --recover_test	Flags::RECOVER_TEST_FLAG
# --no_rescue		Flags::NO_RESCUE_FLAG			Do not rescue damaged files
# --fetch_only		Flags::FETCH_ONLY_FLAG			Stop after fetching the ccsds files
# --fits_only		Flags::FITS_ONLY_FLAG			Stop after making the fits files (no metrics)
# --no_soda		Flags::NO_SODA_FLAG			Do not copy fits files to soda
# --skip_hk		Flags::SKIP_HK				Don't do the hk covergae check
# --old_plan		Flags::OLD_SEQ_FLAG			Do not adjust for sequence count overflow
# --special		Flags::SPECIAL = "special"		Special tests
# --louisa		Flags::NO_SODA_FLAG,			For Louisa's non-compressed files
#			Flags::SPECIAL = "no_soda",
#			Flags::USE_NEW_RESCUE reset
# --ignore_satrot	Flags::IGNORE_SATROT			Ignore gaps in the sat rot coverage
#
# Flags::SPECIAL defaults to the empty string
#
# Switches appear in any order before the start and end dates, eg:
#    daily_merge_mission1 [switches] --sdate=yyyymmdd --edate=yyyymmdd --stime=hhmm [--etime=hhmm]
#

# Check switches and set flags.
# Distinguish between interactive start and start by cron, setting all dates appropriately.
Args::arg_check(@ARGV);

# Check whether the status data is available for the duration of this reformat.
&$System::run_stage(\&Hk::check_availability, "Hk::check_availability : Status coverage check");

# Check whether the sat rot csv files are up to date                                                                    
&$System::run_stage(\&Hk::check_sat_rot, "Hk::check_sat_rot : Check whether the sat rot csv files cover reformat period");

goto reformat_fits if $Flags::JUST_FITS_FLAG;
goto skip_fetch    if $Flags::RECOVER_TEST_FLAG;

# Remove existing mission data files if any, fetch new data and check integrity of received files.
&$System::run_stage(\&Data::clean, "Data::clean : Remove existing data");
&$System::run_stage(\&Data::fetch, "Data::fetch : Fetch data and place in the received directory ($Filesystem::MERGE_DIR)");
&$System::run_stage(\&Data::check, "Data::check : Mission data check");
    
goto the_exit if $Flags::FETCH_ONLY_FLAG;

 skip_fetch:

# Decompress the mission data.
&$System::run_stage(\&Data::decompress, "Data::decompress : Decompress mission data");

 reformat_fits:

# Convert to fits format.
&$System::run_stage(\&Data::reformat, "Data::reformat : Mission data reformat");

goto skip_rescue unless $Flags::RESCUE_MOD;
goto skip_rescue if $Flags::NO_RESCUE_FLAG;

# Rescue damaged data, decompress and reformat into fits.
&$System::run_stage(\&Data::rescue, "Data::rescue : Rescue damaged files");

 skip_rescue:

goto the_exit if $Flags::FITS_ONLY_FLAG;
goto the_exit if $Flags::JUST_FITS_FLAG;

# Update system logs, produce files holding information about the reformated files and compress all the resulting fits files.
&$System::run_stage(\&Data::update_translation_databases, "Data::update_translation_databases : Update md translation database");
&$System::run_stage(\&Data::fits_header_dump, "Data::fits_header_dump : Generating fits header dump");
&$System::run_stage(\&Data::blank_pixel_check, "Data::blank_pixel_check : Generating blank pixel data");
&$System::run_stage(\&Data::check_table_dump, "Data::check_table_dump : Generating check table dump");
&$System::run_stage(\&Data::compress_fits, "Data::compress_fits : Gzip fits files");

goto the_exit         if $Flags::RECOVER_TEST_FLAG;

# Move engineering fits files to the engineering area (not SODA).
# Move science fits files to SODA.
&$System::run_stage(\&Mission::populate_mission_data, "Mission::populate_mission_data : Moving mission data files to soda");

goto skip_quicklook if $Flags::TESTING;

# Remove quicklook fits files (if any) as they have been superceded.
&$System::run_stage(\&Mission::remove_quicklook, "Mission::remove_quicklook : Removing quicklook mission data for $Dates::start_date_string");

 skip_quicklook:
 skip_soda_update:

# Summary files for the pipeline process.
&$System::run_stage(\&Mission::generate_summary_files, "Mission::generate_summary_files : Generating summary files");

 the_exit:

# Leave only footprints and take only memories.
#Mission::clean_up();
Pipeline::pipeline_exit(0, "Finished normally");
