#!/usr/bin/env perl

# M. Whillock, m.whillock@ucl.ac.uk, MSSL, UCL, 2006-2024

# cron start
# ./daily_merge_status_v2.pl [v1|v2]
# ./daily_merge_status_v2.pl -c [v1|v2]
# ./daily_merge_status_v2.pl -p [v1|v2] '20200604' '20200605'

# interactive start
# ./daily_merge_status_v2.pl [v1|v2] '20200604' '20200605'

use strict;
use warnings;

use Trace::Trace;

# The used libraries pull in the other libraries so we do not have to
# specify all of them here...

use Hinode::EIS::EisPipeline::Args;
use Hinode::EIS::EisPipeline::Data;
use Hinode::EIS::EisPipeline::Pipeline;
use Hinode::EIS::EisPipeline::System;

use Hinode::EIS::EisPipeline::EisStatus::Data;
use Hinode::EIS::EisPipeline::EisStatus::Fits;
use Hinode::EIS::EisPipeline::EisStatus::Flags;

Trace::trace_on();
#Trace::save_to_file("/Users/mcrw/tmp/status_trace.txt");
Trace::save_to_file($Filesystem::STATUS_TRACE_FILE);

System::set_timezone_string("(JST)");

# Creates and initialises the status data pipeline
Pipeline::create("Status");

#
# Flag REFORMAT_VERSION defaults to "v1". Options are "v1", "v2"
#
# Optional command line switches and the flags they set to 1 (reset (0) by default) are:
#   -c			Flags::cron				Started by cron
#   -p			Flags::cron, Flags::cron_pending	Started by cron from pending file
#   TEST		Flags::TESTING				General purpose testing
#   just_fits		Flags::JUST_FITS_FLAG			Don't fetch new data, reformat existing data
#   fetch_only		Flags::FETCH_ONLY_FLAG			Only fetch new data, do not reformat into fits
#   fits_only		Flags::FITS_ONLY_FLAG			Don't fetch new data, reformat existing ccsds files
#   no_update		Flags::NO_UPDATE_DARTS_TRENDS		Don't update qcm or trend files
#   v1                  Flags::REFORMAT_VERSION			Set to v1, performs v1 style reformat
#   v2                  Flags::REFORMAT_VERSION			Set to v2, performs v2 style reformat (default if unspecified)
#   special		Flags::SPECIAL = "special"		Special tests
#   DPRINT		System::DEBUG_PRINT			Enables debug printing
#   trace
#   verbose
#
# Flags::SPECIAL defaults to the empty string
#
# Switches appear in any order before the start and end dates, eg:
#    daily_merge_status [switches] [start_date end_date]
#

# Check switches and set flags.
# Distinguish between interactive start and start by cron, setting all dates appropriately.
Args::arg_check(@ARGV);

goto check_only if $Flags::CHECK_ONLY_FLAG;
goto skip_fetch if $Flags::JUST_FITS_FLAG;

# Remove existing status, dump, mission data files and fits files if any, fetch new data and check integrity of received files.
&$System::run_stage(\&Data::clean, "Data::clean : Remove existing data if any");

goto skip_fetch if $Flags::NO_FETCH_FLAG;

# Fetch new data
&$System::run_stage(\&Data::fetch, "Data::fetch : Fetch new status ccsds packets");

# Change this so the exit reason gets to the exit routine...
goto the_exit      if $Flags::FETCH_ONLY_FLAG;

goto skip_check    if $Flags::NO_CHECK;

# Check the new data
&$System::run_stage(\&Data::check, "Data::check : Check the status data");

 skip_check:

&$System::run_stage(\&Data::prepare_timing_files, "Data::prepare_timing_files : Split status timing files into types");

 skip_timing:
    
 skip_fetch:

 reformat_fits:

# Convert to fits format.
&$System::run_stage(\&Data::reformat, "Data::reformat : Status data reformat");

goto the_exit if (($Flags::FITS_ONLY_FLAG) && (&$System::comment("fits_only flag set, skipping to the exit")));

# Daily plots
&$System::run_stage(\&Fits::daily_plots, "Fits::daily_plots : Create the daily plots");

 check_only:

# Checks
&$System::run_stage(\&Fits::daily_check, "Fits::daily_check : Create the check files");

goto the_exit if $Flags::CHECK_ONLY_FLAG;

# Trends
&$System::run_stage(\&Fits::monthly_trends, "Fits::monthly_trends : Update the monthly trends files");

# Compress fits files
&$System::run_stage(\&Fits::compress_files, "Fits::compress_files : Compress the status fits files");

# Move fits to DARTS
&$System::run_stage(\&Fits::populate_darts, "Fits::populate_darts : Move the fits files to DARTS");

# Get sequence counts and move ccsds to DARTS
&$System::run_stage(\&Data::process_ccsds_files, "Data::process_ccsds_files : Get sequence count info and move the ccsds status files and dmp files (if any) to DARTS");

# Remove status quicklook if any
&$System::run_stage(\&Fits::remove_quicklook, "Fits::remove_quicklook : remove quicklook status data for this date");

 the_exit:

# Leave only footprints and take only memories.
###Pipeline::clean_up();

#Pipeline::exit_pipeline(0, "Finished normally");
Pipeline::pipeline_exit(0, "Finished normally");
