#!/bin/bash

source ${HOME}/bin/pipeline_includes.sh

#
# allband = 0xF1
# antenna = 0xF0 merge
#           0xF1 allant
#

###use Env qw(EGSE_SW_LIB);

# Instead of die-ing would be better to report the error in a file somewhere
# and then exit. This applies to all instances of die of course. Todo.

#do "$EGSE_SW_LIB/date_calc.pl"     || die "Can't do date_calc.pl: $!\n";
#do "$EGSE_SW_LIB/pipeline_defs.pl" || die "Can't do pipeline_defs.pl: $!\n";

declare -i debug=1;

#my %trans = ();
#my %temp = ();
#my %rtrans = ();

# Open the master pipeline log if we can.
###PIPELINE_LOG="$HOME/work/localdata/log/pipeline_log.txt"
#open(MLOG, ">> $pipeline_log/pipeline_log.txt") || die "Can't open $pipeline_log/$pipeline_log.txt for writing: $!";
#print MLOG scalar(localtime), " (JST) daily_merge_mission2 started ";

#date_today=`date`
##echo -n `date` " (JST) daily_merge_mission2 started " >> $PIPELINE_LOG
#echo -n ${date_today} daily_merge_mission2 started >> ${PIPELINE_LOG}

#announce("daily_merge_mission2")

# The start time will either be passed in (interactive mode)
# or will be generated from cron using todays date minus some days

#$state = "cron"; # Default to cron job

#$ant='usc34';	# Not needed for merge as sdtp will substitute merge antenna
#$band=3;	# Merge telemetry
#$mode='merge';	# Merge telemetry

# Set default values for start date, end date and start time to empty strings.
# If in interactive state these are not set then bail out.

sdate=""
edate=""
stime=""
etime=""

# If the command line is empty then started by cron
if [ -z $@ ]; then
#    echo "by cron" >> ${PIPELINE_LOG}
    announce("daily_merge_mission2", "by cron")

    init_pipeline("cron")

    # Started by cron. Calculate start date, end date and start time

#    @today = today();	# @today[0..6] = seconds,minutes,hours,day,month,year,day_of_week (0 = sunday)

###    tday=`date +%a`	# Gives the day (i.e. Thu)

    tday=$(date)	# Gives the full date (i.e. Thu 30 Jun 2011 12:02:22 BST

###    day_name=$(echo $tday | cut --fields=0-0)

    day_name=`date +%a`

    # If today is sunday then don't bother to do anything
    if [ ${day_name} = Sun ]; then
	echo Sunday - not running
	pipeline_exit("mission")
    fi

    declare -i start_day_offset=7
    declare -i end_day_offset=`expr $start_day_offset - 1`
	
    # If today is saturday then do saturday until monday morning
    if [ ${day_name} = Sat ]; then
	end_day_offset=`expr ${end_day_offset} - 1`
#	(($end_day_offset++))
    fi

    # Save defaults for start and end dates in case we can't get/parse orl files
    default_start_day=$(date --date="${tday} ${start_day_offset} days ago" +%Y%m%d)
    default_end_day=$(date --date='${tday} ${end_day_offset} days ago' +%Y%m%d)

    # Extract strings from the dates
    default_syear=$(${default_start_day}:0:4)
    default_smonth=$(${default_start_day}:4:2)
    default_sday=$(${default_start_day}:6:2)

    default_eyear=$($default_end_day:0:4)
    default_emonth=$($default_end_day:4:2)
    default_eday=$($default_end_day:6:2)

    # Form start date string
    default_sdate=${default_start_day}
    default_edate=${default_end_day}

    # Now try to parse start day orl file. If orl file not found for today, 
    # try tomorrow and lastly 3 days hence.
    # If all that fails then use today 10:00

    start_time_from_orl=0;
    echo "	Looking for start time in orl file for ${default_sdate}" >> ${PIPELINE_LOG}
    @start_day = other_date(@today, ${start_day_offset});		# Go back 7 days

    #
    # Extract strings from the dates
    #
    $syear  = sprintf "%04u", $start_day[5];	# Get starting year
    $smonth = sprintf "%02u", $start_day[4];	# Get starting month
    $sday   = sprintf "%02u", $start_day[3];	# Get starting day
    
    #
    # Form start date string
    #
    sdate="${syear}${smonth}${sday}"
    
    #
    # Parse orl file
    #
    stime_str=parse_orl_file(${sdate})
    
    if((${stime_str} ne "not found") && (${stime_str} ne "parse error")) {
	${start_time_from_orl}=1
    }
    elsif(${stime_str} eq "not found") {
	MLOG("\torl file not found. Quitting.")
	pipeline_exit("mission");
    }
    elsif($stime_str eq "parse error") {
	MLOG("\torl file parse error. Quitting.")
	pipeline_exit("mission");
    }

    #
    # Now try to parse end day orl file. If orl file not found for today,
    # try tomorrow and lastly 3 days hence.
    # If all that fails then use today+1 10:00
    #
    end_time_from_orl=0
    for attempt in $(seq 2) ; do
#    foreach $attempt (0 .. 2) {
	end_day_offset=((${start_day_offset} + 1))
#	print "end_day_offset = $end_day_offset\n";
	@end_day   = other_date(@today, $end_day_offset);		# Go back 5 or 6 days

	#
	# Extract strings from the dates
	#
	$eyear  = sprintf "%04u", $end_day[5];	# Get ending year
	$emonth = sprintf "%02u", $end_day[4];	# Get ending month
	$eday   = sprintf "%02u", $end_day[3];	# Get ending day
	MLOG("\tLooking for end time in orl file for ${eyear}${emonth}${eday}")

	#
	# Form end date strings
	#
	edate="${eyear}${emonth}${eday}"
    
	#
	# Parse orl file
	#
	etime_str=parse_orl_file(${edate}, *MLOG);

	if(($etime_str ne "not found") && ($etime_str ne "parse error")) {
	    $end_time_from_orl = 1;
	    last;
	}
	$start_day_offset += 1;

    }

    if(! $start_time_from_orl) {
	print MLOG "\tError parsing orl file - using default (1000) for start time\n";
	$stime_str = "1000";
    }
    else {
	print MLOG "\tStart time from orl file - $stime_str\n";
    }

    if(! $end_time_from_orl) {
	print MLOG "\tError parsing orl file - using default (1000) for end time\n";
	$etime_str = "1000";
    }
    else {
	print MLOG "\tStop time from orl file - $etime_str\n";
    }

    if(! $start_time_from_orl) {
	print MLOG "\tNo start orl file found, bailing\n";
	pipeline_exit("mission");
    }

    if((! $start_time_from_orl) || (! $end_time_from_orl)) {
	#
	# Extract strings from the dates
	#
	$syear  = sprintf "%04u", $default_start_day[5];	# Get starting year
	$smonth = sprintf "%02u", $default_start_day[4];	# Get starting month
	$sday   = sprintf "%02u", $default_start_day[3];	# Get starting day

	$eyear  = sprintf "%04u", $default_end_day[5];	# Get ending year
	$emonth = sprintf "%02u", $default_end_day[4];	# Get ending month
	$eday   = sprintf "%02u", $default_end_day[3];	# Get ending day

	#
	# Form start date string
	#
	$sdate = "$syear$smonth$sday";
	$edate = "$eyear$emonth$eday";

	$stime_str = "1000";
	$etime_str = "1000";
    }
    
    #
    # Open the local log if we can
    #
    pipeline_mission_log="${pipeline_log}/pipeline_mission_log_${sdate}"
#    $pipeline_mission_log = "$pipeline_log/pipeline_mission2_log_$sdate" if $debug;

    open(LOG, ">> $pipeline_mission_log") || die "Can't open $pipeline_mission_log for writing: $!";
    print LOG "Started ", scalar(localtime), " (JST)\n";
    print LOG "\tStart time $stime_str on $sdate\n";
    print LOG "\tEnd   time $etime_str on $edate\n";

    #
    # Addition when using parse_orl
    #
    stime=${stime_str}
    etime=${etime_str}

    #
    # Update master log and off we go
    #
    MLOG("\tdaily_merge_mission2 ${sdate} ${edate} ${stime} ${etime}")

else
    init_pipeline("interactive")

    # Get any command line arguments (must be in this order)
    if(@ARGV) { $sdate  = shift; }
    if(@ARGV) { $edate  = shift; }
    if(@ARGV) { $stime 	= shift; }
    if(@ARGV) { $etime 	= shift; }
    #
    # Simple check for arguments.
    # Could test actual times but sdtp will catch errors
    #
    die "No start date" if $sdate eq "";
    die "No end date"   if $edate eq "";
    die "No start time" if $stime eq "";
    $etime = $stime     if $etime eq "";
    #
    # All set, open the local log, update the master log and off we go
    #
    $pipeline_mission_log = "$pipeline_log/pipeline_mission_log_$sdate";
    $pipeline_mission_log = "$pipeline_log/pipeline_mission2_log_$sdate" if $debug;
    open(LOG, ">> $pipeline_mission_log") || die "Can't open $pipeline_mission_log for writing: $!";
    print LOG "Started (manually) ", scalar(localtime), " (JST)\n";
    print MLOG "manually: daily_merge_mission2 $sdate $edate $stime $etime\n";

    $syear  = substr($sdate, 0, 4);
    $smonth = substr($sdate, 4, 2);
    $sday   = substr($sdate, 6, 2);
fi


###die;

# Split the start date into constituent parts
year=$($sdate:0:4)
month=$($sdate:4:2)
day=$($sdate:6:2)

date_string="${year}${month}${day}"

init_log_names(${debug});
init_local_logs(${date_string}, ${year}, ${month}, ${day})

#goto skip;
#die;

############################################### 0 - CLEAR OLD DATA

stage_title("Removing old data in ${merge_dir}")

# Remove old ccsds packets
perform("cd ${merge_dir} && /bin/rm -f eis_md_*");
perform("cd ${merge_dir} && /bin/rm -f eis_sts*");
perform("cd ${merge_dir} && /bin/rm -f eis_dmp*");

# Remove old decompressed ccsds packets
perform("cd ${merge_dir}/decompressed && /bin/rm -f eis_md_2*");

# Remove old text files
perform("/bin/rm -f ${merge_dir}/*.txt");

#perform(*LOG, $state, "unset noclobber");
perform("unset noclobber");


############################################### 1 - FETCH DATA

stage_title("Fetching data");

perform("${SDTP} ${mode} ${ant} band=${band} sdate=${sdate} edate=${edate} stime=${stime} etime=${etime} >& ${SDTP_FETCH_LOG}");


############################################### 2 - MISSION DATA CHECK

stage_title("Mission data check");

ccsds_files=`ls ${merge_dir}/ | grep eis_md`
declare -i total_ccsds_files=${#ccsds_files}
if [ ${total_ccsds_files} eq 0 ]; then
    log_msg("No ccsds mission data files")
    pipeline_exit("mission")
fi

log_msg("Number of files: ${total_ccsds_files}")

# Do the mission data header dump, which also updates the engineering.txt file
perform("$HOME/bin/pipeline_md_hdr_dump $merge_dir > $MD_HDR_CHECK_LOG")

# Do the ccsds check which detects missing packets, updating missing_packets.txt
perform("$HOME/bin/pipeline_ccsds_check $merge_dir > $CCSDS_PACKET_CHECK_LOG")

# Do later
# Move the files to DARTS
#perform(*LOG, $state, "/bin/mv $merge_dir/md_hdr_check.txt $darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html");
#perform(*LOG, $state, "/bin/mv $merge_dir/ccsds_hdr_check.txt $darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt");
#perform(*LOG, $state, "/bin/cp $shutter_log/shutter.txt $staging/logs/");

# If there are any archives with missing packets move them to the rescue directory for processing later on
declare -i number_missing=0;
if [ -e $MISSING_PACKETS_FILE ]; then
    log_msg("Missing packet file exists");
    while read line; do
        # skip header
        [[ ${line:0:7} == "Missing" ]] && continue
        # skip empty lines
        [[ -z "$line" ]] && break
	perform("/bin/mv $MERGE_DIR/$word0 $merge_dir/rescue/");
	(($number_missing++))
    done < $MISSING_PACKETS_FILE
    perform("/bin/mv $merge_dir/missing_packets.txt $darts_mission/$year/$month/packet_check/missing_packets.$year$month$day.txt")
    perform("/bin/mv $MISSING_PACKETS_FILE $MISSING_PACKETS_FILE_DARTS")
    log_msg("Number of files missing packets: $number_missing")
else 
    log_msg("No missing packet file")
fi

# Check for and record those files which are continuations of previous files.
# The decompression stage will move these files aside.
declare -i number_headless=0
if [ -e $HEADLESS_PACKET_FILES ]; then
    log_msg("Headless packets file exists")
    while read line; do
        # skip header
        [[ ${line:0:7} == "Headless" ]] && continue
        # skip empty lines
        [[ -z "$line" ]] && break
	perform("/bin/mv $MERGE_DIR/$word0 $merge_dir/rescue/");
	(($number_headless++))
    done < $HEADLESS_PACKET_FILES
    perform(*LOG, $state, "/bin/mv $merge_dir/headless_packets.txt $darts_mission/$year/$month/packet_check/headless_packets.$year$month$day.txt")
    log_msg("Number of files missing packets: $number_headless")
else
    log_msg("No headless packets file")
fi

# Check for and record those files which are not complete.
# The decompression stage will move these files aside.
declare -i number_tailless=0
if [ -e $TAILLESS_PACKET_FILES ]; then
    log_msg("Tailless packets file exists")
    while read line; do
        # skip header
        [[ ${line:0:7} == "Tailless" ]] && continue
        # skip empty lines
        [[ -z "$line" ]] && break
	perform("/bin/mv $MERGE_DIR/$word0 $merge_dir/rescue/");
	(($number_tailless++))
    done < $TAILESS_PACKET_FILES
    perform(*LOG, $state, "/bin/mv $merge_dir/incomplete_packets.txt $darts_mission/$year/$month/packet_check/incomplete_packets.$year$month$day.txt")
    log_msg("Number of files missing packets: $number_tailless")
else
    log_msg("No tailless packets file")
fi

# Move the sequence count continuity file to DARTS
perform("/bin/mv $sequence_counts_file $darts_sequence_counts_file")

# Get a count of number of mission data files here
pre_decompression=`ls $merge_dir/ | grep eis_md`
declare -i total_compressed_files=${#pre_decompression}
log_msg("Number of compressed files: $total_compressed_files")


############################################### 3 - MISSION DATA DECOMPRESS

stage_title("Starting mission data decompress")
(
cat <<'EOF'

.comp $local_idl/eis_ccsds_interface__define

;.comp $local_idl/eis_md_decomp
;.comp $local_idl/eis_md_decomp_script
;eis_md_decomp_script,'$merge_dir', /merge

.comp $local_idl/mdppmdctrecover
;;;.comp $local_idl/mdppmdctrecover_merge

;.comp $local_idl/eis_md_decomp_fix
;.comp $local_idl/eis_md_decomp_fix_script
;eis_md_decomp_fix_script,'$merge_dir', /merge

.comp $local_idl/eis_md_decomp_fix1
.comp $local_idl/eis_md_decomp_fix1_script
eis_md_decomp_fix1_script,'$merge_dir', /merge
EOF
) > "$TEMP_IDL/md_decomp.pro"

# Start the mission data decompress
perform("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_decomp.pro > $md_decompress_log")

# Copy the decompression record and logs to DARTS
perform("/bin/cp $HOME/work/localdata/log/decompression/merge_decomp_record.txt $darts_staging/logs/decompression/")

# Slim the original log file down, then delete it
perform("$HOME/bin/slim.pl < $md_decompress_log > $md_decompress_log_slim")
perform("/bin/rm $md_decompress_log")

# Get a count of number of mission data files here
post_decompression=`ls $merge_dir/decompressed/ | grep eis_md`
declare -i total_decompressed_files=${#post_decompression}
log_msg("Number of de-compressed files: $total_decompressed_files")


############################################### 4 - MISSION DATA REFORMAT

stage_title("Starting mission data reformat")

log_msg("(IDL) fits_script, '$merge_dir/decompressed/', '$temp_fits/")

(
cat <<'EOF'
.comp $local_idl/planning/jside_input/eis_read_orbit_events_reformatter
;;;;;;.comp $local_idl/eis_fits_calib
;;;.comp $local_idl/hw_help__define
;;;;;;.comp $local_idl/eis_data__define
;.comp $local_idl/eis_data__readccsds
;;;.comp $local_idl/eis_data__readfits
;;;.comp $local_idl/eis_modfits
;;;.comp $local_idl/eis_mkfits
;;;.comp $local_idl/eis_fits_script
eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/', logfile='$translate_log'
;eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/',logfile='$HOME/work/localdata/log/merge/md_translation.txt'
;eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/',logfile='$HOME/work/localdata/log/merge/md_translation.txt', /nospcd
EOF
) > "$TEMP_FITS/md_fits.pro"

perform("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log")
perform("/bin/cp $reformat_log $darts_staging/logs/md_reformat/")
perform("/bin/cp $translate_log $darts_staging/logs/md_translation/")

# Get a count of number of mission data files here
post_reformat=`ls $temp_fits/ | grep eis_l0`
declare -i total_reformatted_files=${#post_reformat}
log_msg("Number of reformatted files: $total_reformatted_files")


############################################### 5 - UPDATE MD TRANSLATION DBASE - make subroutine

stage_title("Update md translation database")

# eis_mkfits writes all the ccsds packet filenames and the corresponding fits filenames processed
# to a text file. Open the text file and transfer the information to a perl database.

update_md_translation(\%trans, $translate_log);

#if(open(REF_LOG, "< $translate_log")) {
#    print STDOUT "Opened $translate_log ok\n" if $state eq "interactive";
#    print LOG  "\tOpened $translate_log ok\n";
#    while(<REF_LOG>) {
#	@files = split;		# Split ccsds filename and fits filename
#	$ccsds = $files[0];
#	$fits = $files[1];
#	$ccsds =~ s/ //g;		# Remove any white space from names
#	$fits =~ s/ //g;
#	chomp $fits;		# Remove newline if any
#	$trans{$ccsds} = $fits;
#	print LOG  "\tUpdating mission_name_database: $ccsds => $fits\n";
#	$temp{$ccsds} = $fits;
#    }
#    close(REF_LOG);
#}
#else {
#    print STDOUT "Can't open $reformat_log: $!\n" if $state eq "interactive";
#    print LOG  "\tCan't open $reformat_log: $!\n";
#}


############################################### 6 - COMPRESS

# Now gzip everything in the temporary fits directory
# before moving to the DARTS directories

stage_title("Gzip temporary fits files")

perform("cd $temp_fits && gzip -f *")

# Generate the local study descriptions file
perform("fits_study_metric.csh $temp_fits >  $HOME/tmp/studies.txt")
perform("sed -e '/SSW/,/Type/d' $HOME/tmp/studies.txt > $HOME/tmp/studies.html")

# Generate the local cal study descriptions file
#fits_cal_study1.pl > ~/tmp/cal_studies.txt


############################################### 7 - COPY ENGINEERING FILES TO CALIBRATION AREA


stage_title("Move engineering files to calibration area")

# If the engineering.txt file exists then there is engineering data to be dealt with.
# Use the ccsds packet filename from the engineering.txt file to look up in the translation
# log to get the fits filename and use the engineering study type information to copy the
# fits file to the calibration directory.

$eng_dbase = 0;
if(-e $engineering_file) {
    log_msg(*LOG, $state, "Engineering summary exists");
    if(open(ENG_LOG, "< $engineering_file")) {
	$line = <ENG_LOG>;					# Read title
	while($line = <ENG_LOG>) {				# Read ccsds packet filename
	    last if $line =~ /^$/;				# Finish on a blank line
	    chop $line;						# Remove \n
	    ($path, $type) = split(/:/, $line);			# Get filename and type of engineering study
	    log_msg(*LOG, $state, "(path,type) = ($path, $type)");
	    @path = split(/\//, $path);				# Split up filename path
	    $ccsds_file = $path[-1];				# Get the ccsds filename
#	    $fits_file = $etrans{$ccsds_file};			# Get fits filename from ccsds filename
	    $fits_file = $temp{$ccsds_file};			# Get fits filename from ccsds filename
	    if($fits_file) {
		$eng_dbase{$fits_file} = $type if $eng_dbase;	# Update engineering database
		$destination = eng_path($type);			# Get where to copy the fits file and copy it
		$source = $temp_fits . "/"  . $fits_file . ".gz";
		if($destination ne "") {
		    $dest1 = "$darts_mission" . $destination;
		    $dest2 = "$soda_darts_mission" . $destination;
		    log_msg(*LOG, $state, "Moving $source to $dest1 and $dest2");
		    # mcrw 25/07/07 - Move instead of copy so engineering files don't appear in science area
		    perform(*LOG, $state, "/bin/cp $source $dest1");
		    perform(*LOG, $state, "/bin/mv $source $dest2");
		}
		else {
		    log_msg(*LOG, $state, "No destination found for $fits_file ($type)");
		}
	    }
	    else {
		log_msg(*LOG, $state, "No translation found for $ccsds_file");
	    }
	}
	close(ENG_LOG);
    }

    perform(*LOG, $state, "/bin/mv $engineering_file $darts_engineering_file");

    #
    # Generate cal study summary file from fits here?
    #  ./fits_cal_study.pl > ~/work/localdata/log/mission/cal_studies_list.txt
    #
}
else {
    log_msg(*LOG, $state, "No engineering summary file");
}


############################################### 8 - MOVE MISSION FITS TO DARTS

stage_title("Moving mission data fits files to DARTS")

move_fits_to_soda();

#
# Filenames:
#	eis_l0_yyyymmdd_hhmmss.fits
#

#@files = `ls $temp_fits`;
#foreach $file (@files) {
#    next if !($file =~ /l0/);
#    chomp $file;
#    # Match destination directory with filename
#    $darts_year  = substr $file,  7, 4;
#    $darts_month = substr $file, 11, 2;
#    $darts_day   = substr $file, 13, 2;
##    perform(*LOG, $state, "/bin/cp $temp_fits/$file $soda_md_dir");
#    perform(*LOG, $state, "/bin/mv $temp_fits/$file $soda_md_dir");
#}


############################################### 10 - REMOVE QUICKLOOK MISSION DATA

stage_title("Removing quicklook mission data for $year$month$day")

# Remove directory in DARTS if it exists
if [ -e "$darts_mission_ql/$year/$month/$day" ]; then
    perform("/bin/rm -rf $darts_mission_ql/$year/$month/$day")
else
    log_msg("\tDirectory $darts_mission_ql/$year/$month/$day does not exist\n")
fi

# Remove md directory in sdtp if it exists
if [ -e "$ql_mission/$year$month$day" ]; then
    perform("/bin/rm -rf $ql_mission/$year$month$day")
else
    log_msg("\tDirectory $ql_mission/$year$month$day does not exist")
fi}

# Remove md fits directory in sdtp if it exists
if [ -e "$ql_mission_fits/$year$month$day"]; then
    perform("/bin/rm -rf $ql_mission_fits/$year$month$day")
else
    log_msg("\tDirectory $ql_mission_fits/$year$month$day does not exist")
fi


############################################### 11 - Rescue

stage_title("Performing rescue")

if [ $number_missing > 0 ]; then
    # Remove old ccsds packets
    perform("cd $merge_dir && /bin/rm -f eis_md_2*")
    # Remove old decompressed ccsds packets
    perform("cd $merge_dir/decompressed && /bin/rm -f eis_md_2*")

    # Perform rescue
    $rescue_pre_rescue=`ls $merge_dir/rescue/eis_md*`
    declare -i $pre_rescue=${#rescue_pre_rescue}
    log_msg("Number of files to rescue: $pre_rescue")

    # Create rescue script
(
    cat << EOF
.comp $local_idl/rescue/eis_ccsds_interface__define
.comp $local_idl/rescue/eis_ccsds_reader__define
.comp $local_idl/rescue/eis_md_interface__define
.comp $local_idl/rescue/eis_md_reader__define
.comp $local_idl/rescue/eis_md_rescue__define
.comp $local_idl/rescue/eis_md_recover

eis_md_recover, /dir, /merge, datestr='$sdate_$edate'
EOF
) > $RESCUE

perform("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/rescue.pro")
rescue_post_rescue=`ls $merge_dir/ | grep eis_md`
declare -i $post_rescue=${#rescue_post_rescue}
log_msg("Number of files rescued: $post_rescue")
if [ $post_rescue -gt 0 ]; then
    for file in $rescue_post_rescue; do
	    # Get file size, if 0 mv source to incomplete
#	    @stat = stat $file;
	    # Remember file size for each rescued file
#	    $rescued_files{$file} = $stat[7];
	if [ -s $file ]; then	# file exists and is not empty
	else
	    perform("mv $merge_dir/rescue/$file $merge_dir/incomplete")
	    perform("rm $merge_dir/$file")
	fi
    done
fi

    # Decompress
rescue_pre_decompression=`ls $merge_dir/ | grep eis_md`
declare -i post_rescue=${#rescue_pre_decompression}
log_msg("Number of non-empty files rescued: $post_rescue")
    
if [ $post_rescue -gt 0 ]; then
    perform("rescue_decomp.pl $sdate_$edate")
fi
    
rescue_post_decompression=`ls $merge_dir/decompressed/ | grep eis_md`
declare -i post_rescue=${#rescue_post_decompression}
log_msg("Number of decompressed files rescued: $post_rescue")

    # Reformat
if [ $post_rescue -gt 0 ]; then
	# Do exposure count on the rescued ccsds files here
    perform("rescue_reformat.pl $sdate_$edate")
fi

rescue_post_reformat=`ls $temp_fits`
declare -i post_rescue=${#rescue_post_reformat}
log_msg("Number of reformatted files rescued: $post_rescue")
if [ $post_rescue -gt 0 ]; then
    update_md_translation(\%rtrans, $rescue_reformat_log)
    perform("cd $temp_fits && gzip -f *")
    perform("fits_study_metric.csh $temp_fits >  $HOME/tmp/_rescued_studies.txt")
    perform("sed -e '/SSW/,/Type/d' $HOME/tmp/studies.txt > $HOME/tmp/rescue_studies.html")
#	perform(*LOG, $state, "/bin/rm $HOME/tmp/rescued_studies.txt");
fi


############################################### 8 - MOVE MISSION FITS TO DARTS

#
# Move rescued fits to soda
#

stage_title("Moving mission data fits files to DARTS")

move_fits_to_soda();


############################################### 12 - Generate summary files
skip:

stage_title("Generating summary files")

# Read in raster exposures file
# @ccsds_files contains a list of all eis_md files
# @pre_decompression
# @post_decompression
# @post_reformat ?

ccsds_missing_file="$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt"
plan_file="$staging/eis_plan/eis_plan_$syear$smonth$sday.txt"
# Engineering file is $darts_mission/$year/$month/packet_check/engineering.$year$month$day.txt

# Estimate the number of rasters expected for this plan
expected_number_of_rasters=`rasters_for_plan.sh $plan_file`

# Generate and reorganize the plan file
perform("estimate_rasters1.pl < $plan_file > $HOME/tmp/daily_plan.txt")

# Get the number of exposures per mission data per packet
# Output format: filename	packets_per_exposure	number_of_exposures	data_volume	timeline_id (SID)	rasters_required	rasters_done	RID	MID
perform("exposure_count.pl < $darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html > $HOME/tmp/packets_per_exposure.txt")

# Generate the list of files with number of good and damaged exposures
# Output format: filename	headers read	headers damaged
perform("exposures_missing_packets.pl < $darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt > $HOME/tmp/exposure_list.txt")

# Generate the file of sequence count continuity
# Output format: Filename    Start    End   Diff   Rec
perform("sequence_count_continuity_check.pl < $darts_mission/$year/$month/packet_check/sequence_counts_$year$month$day.txt > $HOME/tmp/cont_check.txt")

# Generate the file of missing ccsds packets summary
# Output format: Filename	received	missing		gaps
perform("ccsds_missing_summary1.pl $ccsds_missing_file > ~/tmp/missing.txt")
declare -i total_ccsds_packets=`awk '{ tot += \$2 } END { print tot }' ~/tmp/missing.txt`
declare -i total_ccsds_packets_missing=`awk '{ tot += \$3 } END { print tot }' ~/tmp/missing.txt`
declare -i total_gaps=`awk '{ tot += \$4 } END { print tot }' ~/tmp/missing.txt`
declare -i total_files=`grep eis_md ~/tmp/missing.txt | wc -l`

#if(open SUMMARY, ">>$HOME/tmp/missing.txt") {
#    my $fits_num = scalar(@post_reformat);
#    print SUMMARY "\n";
#    print SUMMARY "Planned: $expected_number_of_rasters\nReceived: $total_files\nCcsds total: $total_ccsds_packets\nMissing: $total_ccsds_packets_missing\nGaps: $total_gaps\nNumber of fits files: $fits_num\n";
#    close SUMMARY;
#}

#
# Open up and read plan summary
#
if(open(PLN, "< $HOME/tmp/daily_plan.txt")) {
    while(<PLN>) {
	chomp;
	@items = split;
	push @tl_entries, [@items];
    }
    close PLN;
}

#
# Open up and read regular merge translation file
#
# Output format: Filename    eis_l0_filename
if(open(TRA, "< $translate_log")) {
    while(<TRA>) {
	chomp;
	($md, $l0) = split;
	$md_translation{$md} = $l0;
	$l0_translation{$l0} = $md;
    }
    close TRA;
}

#
# Open up and read rescue translation file
#
# Output format: Filename    eis_l0_filename
if(open(TRA, "< $pipeline_log/md_translation_rescue_$sdate_$edate.txt")) {
    while(<TRA>) {
	chomp;
	($md, $l0) = split;
	$rescue_md_translation{$md} = $l0;
	$rescue_l0_translation{$l0} = $md;
    }
    close TRA;
}

#
# Open up and read raster database dump to get number of exposures per raster
# File format: id acronym type num_exposures num_exposures_per_step
#
# Output format: Id	Acronym		type	nExps	ExpsPerPacket
if(open(RA_DB, "< $raster_db_dump")) {
    $line = <RA_DB>;					# Read title
    while(<RA_DB>) {
	chomp;
	@list = split;
	$rasters{$list[1]} = @list;
    }
    close(RA_DB);
}

#
# Open up and read packets per exposure file
#
# Output format: filename	packets_per_exposure	number_of_exposures	data_volume	timeline_id (SID)	rasters_required	rasters_done	RID	MID
if(open(PPE, "< $HOME/tmp/packets_per_exposure.txt")) {
    while(<PPE>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close PPE;
}

#
# Open up and read sequence continuity check
#
# Output format: Filename    Start    End   Diff   Rec
if(open(CC, "< $HOME/tmp/cont_check.txt")) {
    while(<CC>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close CC;
}

#
# Open up and read exposure list file
#
# Output format: filename	headers read	headers damaged
if(open(EL, "< $HOME/tmp/exposure_list.txt")) {
    while(<EL>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close EL;
}

#
# Open up and read missing list
#
# Output format: Filename	received	missing		gaps
if(open(MIS, "< $HOME/tmp/missing.txt")) {
    while(<MIS>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close MIS;
}

#
# Open up and read main study list
#
if(open(MSL, "< $HOME/tmp/studies.txt")) {
    while(<MSL>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close MSL;
}

#
# Open up and read rescued study list
#
if(open(MSL, "< $HOME/tmp/rescued_studies.txt")) {
    while(<MSL>) {
	chomp;
	@list = split;
	$ppe{$list[0]} = @list;
    }
    close MSL;
}

#
# Make output file here
#
if(open(OUT, "> $HOME/tmp/reformat_report.txt")) {
    print OUT "+table\n+name\nSummary\n-name\n";
    print OUT "+supertitles\n";
    print OUT "Study 3\n";
    print OUT "Raster 7\n";
    print OUT "Files 2\n";
    print OUT "Ccsds 3\n";
    print OUT "-supertitles\n";
    print OUT "+titles\n";
    print OUT "Timeline Id\nStudy Id\nStudy Acronym\nRaster Id\nRaster Acronym\nPlanned Repeats\nRepeat Number\nStatus\nExpected Exposures\nActual Exposures\n";
    print OUT "Fits File\nSource File\nReceived\nMissing\nGaps\n";
    print OUT "-titles\n";

    $previous_tl_id     = 0;
    $previous_raster_id = 0;

    foreach $tl_entry (@tl_entries) {
	@items = @{$tl_entry};
	$num_reps = $items[14];
	$tl_id = $items[1];
	$repeat_number = 1;
	$raster_id = $items[6];
	if($tl_id == $previous_tl_id) {
	    if($raster_id == $previous_raster_id) {	# continuation raster of raster repeat
		print OUT "+items\n \n \n \n$items[6]\n$items[7]\n \n$repeat_number\nOK\n10\n10\neis_l0\neis_md\n1000\n0\n0\n-items\n";
	    }
	    else {	# new raster of a study
		print OUT "+items\n \n \n \n$items[6]\n$items[7]\n$items[14]\n$repeat_number\nOK\n10\n10\neis_l0\neis_md\n1000\n0\n0\n-items\n";
	    }
	    ++$repeat_number;
	}
	else {
	    print OUT "+entry\n";
	    if($num_reps > 1) {
		print OUT "+items\n$items[1]\n$items[3]\n$items[4]\n$items[6]\n$items[7]\n$items[14]\n$repeat_number\nOK\n10\n10\neis_l0\neis_md\n1000\n0\n0\n-items\n";
		for($i = 1; $i < $num_reps; ++$i) {
		    ++$repeat_number;
		    print OUT "+items\n \n \n \n$items[6]\n$items[7]\n \n$repeat_number\nOK\n10\n10\neis_l0\neis_md\n1000\n0\n0\n-items\n";
		}
		print OUT "-entry\n";
	    }
	    else {
		print OUT "+items\n$items[1]\n$items[3]\n$items[4]\n$items[6]\n$items[7]\n$items[14]\n$repeat_number\nOK\n10\n10\neis_l0\neis_md\n1000\n0\n0\n-items\n";
	    }
	}
	$previous_tl_id = $tl_id;
	$previous_raster_id = $raster_id;
    }
    print OUT "-entry\n";
    print OUT "-table\n";
    close OUT;
}

perform("make_tables.pl $year$month$day < $HOME/tmp/reformat_report.txt > $darts_staging/reformat_summary/$year$month$day.html")
#perform(*LOG, $state, "/bin/mv ~/tmp/missing.txt $staging/packet_summary/summary_$syear$smonth$sday.txt");
#die;

perform("study_log.csh $syear $smonth")


############################################### 9 - MOVE LOGS TO DARTS

stage_title("Moving log files to DARTS")

# Move the files to DARTS
#perform(*LOG, $state, "/bin/mv $merge_dir/md_hdr_check.txt $darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html");
#perform(*LOG, $state, "/bin/mv $merge_dir/ccsds_hdr_check.txt $darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt");
#perform(*LOG, $state, "/bin/cp $shutter_log/shutter.txt $staging/logs/");


############################################### 9 - CLEAN UP

stage_title("Cleaning up");

# Remove temporary idl programs
perform("/bin/rm $temp_idl/*pro")

# Remove temporary fits files
#perform(*LOG, $state, "cd $temp_fits && /bin/rm eis_l0*");

# Remove any temporary files from decompression
#print LOG "\t/bin/rm $eis_local/tmp/eis_md*\n";
###system("/bin/rm $eis_local/tmp/eis_md*");

perform("mv $HOME/tmp/merge_record.txt $HOME/data/staging")
perform("gzip -f $HOME/data/staging/merge_record.txt")


############################################### 13 - EXIT

stage_title("Exiting")
perform("set noclobber")
pipeline_exit("mission")


############################################### subs


sub parse_contact_file {
    my $date = shift;
    my $fh   = shift;
    my $line;
    my $src_file;
    my $file;
	
    # Parse contact file to get start time
    $src_file = "$date" . "_contacts.txt";
    $file = `/bin/ls $pass_list_dir/$src_file`;
    if($file eq "") {
	print $fh "\tCan't find contact file for $date\n";
	return "";
    }
    chomp $file;
    print $fh "\tUsing contact file $file for $date\n";
    $line = "";

    # Get the second real time contact if it exists
    # otherwise get the first
    if(open(PASS, "< $file")) {
    	while(<PASS>) {
	    if(/real/) {
		chomp;
		$line = $_;
###		last;
	    }
    	}
    }
    else {
    	print $fh "\tCan't open contacts file for $date ($!)\n";
    	return "";
    }
    if($line eq "") {
        print $fh "\tNo start time found in $file\n";
        return "";
    }
    return $line;
}

function parse_orl_file {
    local line
    local src_file
    local file
	
    # Parse orl file to get start time. $orl_dir is defined in pipeline_defs.
    src_file="EIS_$1????.orl"
    file=`/bin/ls $orl_dir/$src_file`
    if [ -z $file ] ; then
	log_msg("Can't find orl file for $1")
	return "not found"
    fi

    log_msg("Using orl file $file for $1")
    line=""

    while read line; do
	[[ ${line#START:OPOG_UPLOAD_TI} ]] && continue
	[[ ${line#START} ]] && break
    done < $file
    if [ -z $line ] ; then
        log_msg("No start time found in $file")
	return "parse error"
    fi

    #
    # $line should be like: START:2007/02/13.10:00:00 { SEQ:EIS_LOAD_OBSTBL_REAL ;}
    # Return hhmm portion
    #
    hh=${line:17:2}
    mm=${line:20:2}

    if [ $hh ==  ${hh##([0-1][0-9]|2[0-3])} ] ; then
	if [ $mm == ${mm##[0-5][0-9]} ] ; then
	    return "$hh$mm"
	fi
    else
	return "parse error"
    fi

#    return "parse error" unless $hh =~ m/([0-1][0-9]|2[0-3])/;
#    return "parse error" unless $mm =~ m/[0-5][0-9]/;

#    return "$hh$mm";
}

# eis_mkfits writes all the ccsds packet filenames and the corresponding fits filenames processed
# to a text file. Open the text file and transfer the information to a perl database.
function update_md_translation {
#    my $dict = shift;
#    my $translate_log = shift;
#
#    if(open(REF_LOG, "< $translate_log")) {
#	log_msg(*LOG, $state, "Opened $translate_log ok");
#	while(<REF_LOG>) {
#	    @files = split;		# Split ccsds filename and fits filename
#	    $ccsds = $files[0];
#	    $fits = $files[1];
#	    $ccsds =~ s/ //g;		# Remove any white space from names
#	    $fits =~ s/ //g;
#	    chomp $fits;		# Remove newline if any
#	    $$dict{$ccsds} = $fits;
#	    $fits_dict{$fits} = $ccsds;
#	    print LOG  "\tUpdating mission_name_database: $ccsds => $fits\n";
#	    $temp{$ccsds} = $fits;
#	}
#	close(REF_LOG);
#    }
#    else {
#	log_msg(*LOG, $state, "Can't open $translate_log: $!");
#    }   
}

function move_fits_to_soda
{

#
# Filenames:
#	eis_l0_yyyymmdd_hhmmss.fits
#

    files=`ls $temp_fits`
    for file in $files ; do
#	if [ ${file:4:2} == "eis_l0" ] ; then # match
	if [ -n ${file#eis_l0} ] ; then # match
#	if [ -z ${file#eis_l0} ] ; then # no match
	# Match destination directory with filename
	    darts_year=${file:7:4}
	    darts_month=${file:11:2}
	    darts_day=${file:13:2}
	    perform("/bin/mv $temp_fits/$file $soda_md_dir/$darts_year/$darts_month/$darts_day")
	fi
    done
}


sub init_log_names {
    my $debug = shift;

    if($debug) {
	#
	# Log locations during processing
	#
	$sdtp_fetch_log         = "$pipeline_log/m2_md_fetch_$date_string.txt";

	$md_hdr_check_log       = "$merge_dir/m2_md_hdr_check.txt";
	$ccsds_packet_check_log = "$merge_dir/m2_ccsds_hdr_check.txt";
	$missing_packets_file   = "$merge_dir/missing_packets.txt";
	$headless_packet_files  = "$merge_dir/headless_packets.txt";
	$tailless_packet_files  = "$merge_dir/incomplete_packets.txt";
	$sequence_counts_file   = "$merge_dir/sequence_counts.txt";
	$engineering_file       = "$merge_dir/engineering.txt";
	
	$shutter_log            = "$hutter_log/shutter.txt";
	$plan_file              = "$darts_staging/eis_plan/eis_plan_$year$month$day.txt";
	$missing_file           = "$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt";
	
	$md_decompress_log      = "$pipeline_log/m2_md_decompress_$date_string.txt";
	$md_decompress_log_slim = "$darts_staging/logs/decompression/m2_md_decompress_$date_string.txt";
	
	$translate_log_name     = "m2_md_translation" . "_$date_string" . ".txt";
	$translate_log          = "$pipeline_log/" . $translate_log_name;
	$reformat_log           = "$pipeline_log/m2_md_reformat_$date_string.txt";
	
	$rescue_decomp_log      = "";
	$rescue_reformat_log    = "";
	$rescue_translate_log   = "";

	$raster_db_dump         = "$HOME/raster_db_dump.txt";
	$summary_file           = "$HOME/tmp/m2_summary.txt";
	#
	# Locations on DARTS
	#
	$darts_md_hdr_check_log       = "$darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html";
	$darts_ccsds_packet_check_log = "$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt";
	$darts_shutter_log            = "$staging/logs/";
	
	$darts_sequence_counts_file   = "$darts_mission/$year/$month/packet_check/sequence_counts_$year$month$day.txt";
	$darts_engineering_file       = "$darts_mission/$year/$month/packet_check/engineering.$year$month$day.txt";
	
	$soda_md_dir                  = "$soda_darts_mission/$darts_year/$darts_month/$darts_day/";
	
	$darts_summary_file           = "$darts_staging/log/reformat_summary/reformat_summary_$year$month$day.html";
    }
    else {
	$sdtp_fetch_log         = "$pipeline_log/md_fetch_$date_string.txt";
	
	$md_hdr_check_log       = "$merge_dir/md_hdr_check.txt";
	$ccsds_packet_check_log = "$merge_dir/ccsds_hdr_check.txt";
	$missing_packets_file   = "$merge_dir/missing_packets.txt";
	$headless_packet_files  = "$merge_dir/headless_packets.txt";
	$tailless_packet_files  = "$merge_dir/incomplete_packets.txt";
	$sequence_counts_file   = "$merge_dir/sequence_counts.txt";
	$engineering_file       = "$merge_dir/engineering.txt";
	
	$shutter_log            = "$hutter_log/shutter.txt";
	$plan_file              = "$darts_staging/eis_plan/eis_plan_$year$month$day.txt";
	$missing_file           = "$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt";
	
	$md_decompress_log      = "$pipeline_log/md_decompress_$date_string.txt";
	$md_decompress_log_slim = "$darts_staging/logs/decompression/md_decompress_$date_string.txt";
	
	$translate_log_name     = "md_translation" . "_$date_string" . ".txt";
	$translate_log          = "$pipeline_log/" . $translate_log_name;
	$reformat_log           = "$pipeline_log/md_reformat_$date_string.txt";
	
	$rescue_decomp_log      = "";
	$rescue_reformat_log    = "";
	$rescue_translate_log   = "";
	
	$raster_db_dump         = "$HOME/raster_db_dump.txt";
	$summary_file           = "$HOME/tmp/summary.txt";
	
	# Locations on DARTS
	$darts_md_hdr_check_log       = "$darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html";
	$darts_ccsds_packet_check_log = "$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt";
	$darts_shutter_log            = "$staging/logs/";
	
	$darts_sequence_counts_file   = "$darts_mission/$year/$month/packet_check/sequence_counts_$year$month$day.txt";
	$darts_engineering_file       = "$darts_mission/$year/$month/packet_check/engineering.$year$month$day.txt";
	
	$soda_md_dir                  = "$soda_darts_mission/$darts_year/$darts_month/$darts_day/";
	
	$darts_summary_file           = "$darts_staging/log/reformat_summary/reformat_summary_$year$month$day.html";
    }
}

