/**************************************
 *
 *  DePacketize
 *
 *    2001/03/26  version 0.2.0 by K.Matsuzaki
 *      created
 *    2001/07/10  version 0.3.0 by K.Matsuzaki
 *      data output change (SUBIMG->CMPIMG)
 *    2001/07/10  version 0.3.0 (replace) by K.Matsuzaki
 *      handle version information created by MDP
 *      added module name in error message 
 *    2001/07/23  version 0.3.1 by K.Matsuzaki
 *      supported multiple instruments proccessing
 *    2001/08/01  version 0.3.3 by K.Matsuzaki
 *      PM integ version
 *    2005/09/24  version 20050920 by K.Matsuzaki
 *      fixed possible inconsistency between header and data part
 *
 **************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <bnkfm.h>
#include <evs.h>
#include <anl.h>

#include <cc-java.h>
#include <solarb/sbstd.h>
#include <solarb/CcsdsPacket.h> /* input */
#include <solarb/MdpPacket.h>   /* output */

#define MODULE "DePacketize:"

/* Interface Descriptions */

#define errlog printf

#define NUM_APID 9

static  byte _arr_verinf  [VERINF_LEN * NUM_APID];   
static  byte _arr_mdphdr  [MDPHDR_LEN * NUM_APID];   
static  byte _arr_datainfo[DATAINFO_LEN * NUM_APID];
static  byte _arr_cmpimg  [CMPIMG_LEN];   

static  byte _packet [NUM_APID][2048];       // !!!!
static  byte _packet1[NUM_APID][2048];       // 20050924
static  byte _imgbuf [NUM_APID][CMPIMG_LEN]; // ~ factor 2 large !!!!


class DePacketize {

#include "DePacketizeBnkIn.cc"
#include "DePacketizeBnkOut.cc"

  DePacketizeBnkIn  _bi;
  DePacketizeBnkOut _bo;

  int _packet_count0     [NUM_APID];

  boolean _stream_close0 [NUM_APID];
  boolean _stream_error  [NUM_APID];
  int     _wrpntr        [NUM_APID];

  PUBLIC void init(){
    _bi = NEW DePacketizeBnkIn();
    _bo = NEW DePacketizeBnkOut();

    _bo.bnkdef();
    _bo.errdef();

    for( int i=0; i<NUM_APID; i++ ){
      _packet_count0[i] = -1; // set 0 for error simulation
      _stream_close0[i] = true;
      _stream_error[i]  = false;
      _wrpntr[i]        =  0;
    }
  }

  void ana(int REF status){
    _bi.bnkget();

    const int     apid         =   _bi.getApid();
    const boolean stream_open  = ((_bi.getSeqnFlag() & 1) != 0);
    const boolean stream_close = ((_bi.getSeqnFlag() & 2) != 0);
    const int     count        =   _bi.getSeqnCount();
    const int     length       =   _bi.getPacketLength();
    const int     ti           =   _bi.getTi();
    const double  sbtime       =   _bi.getSbTime();

    /* bnk (input) */

    int id;
    switch(apid){
    case SB_APID_FLT_OBS_1: id=0; break; 
    case SB_APID_FLT_OBS_2: id=1; break; 
    case SB_APID_SPP_OBS_1: id=2; break; 
    case SB_APID_SPP_OBS_2: id=3; break;  
    case SB_APID_FPP_CT:    id=4; break;
    case SB_APID_XRT_OBS_1: id=5; break;
    case SB_APID_XRT_OBS_2: id=6; break;
    case SB_APID_EIS_OBS_1: id=7; break;
    case SB_APID_EIS_OBS_2: id=8; break;
    default:
      status = ANL_SKIP;
      return;
    }

    status = ANL_OK;

#define COUNTUP(x) (((x)+1)&0x3fff)

    /* == Sequence Conuter Check (connection level) == */
    if( count != COUNTUP(_packet_count0[id]) ){
      EvsSet( MODULE "ERR:SEQN" );
      errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
      errlog( MODULE " packet sequence counter skiped (%d->%d) for APID=%03x\n", _packet_count0[id], count, apid);
    }

    if ( stream_open ){
      int datatype =   _bi.getDatatype();

      int verinf_len   = VERINF_LEN;
      int mdphdr_len   = MDPHDR_LEN;
      int datainfo_len;
      switch(apid){
      case SB_APID_FLT_OBS_1: case SB_APID_FLT_OBS_2: 
      case SB_APID_SPP_OBS_1: case SB_APID_SPP_OBS_2:  
	datainfo_len = FPP_DATAINFO_LEN; 
	break;
      case SB_APID_FPP_CT:
	switch(datatype){
	case DATA_TYPE_CT_REF: datainfo_len = 32; break;
	case DATA_TYPE_CT_LIV: datainfo_len = 32; break;
	case DATA_TYPE_CT_RES: datainfo_len = 64; break;
	}
	break;
      case SB_APID_XRT_OBS_1: case SB_APID_XRT_OBS_2: 
	datainfo_len = XRT_DATAINFO_LEN; 
	break;
      case SB_APID_EIS_OBS_1: case SB_APID_EIS_OBS_2: 
	datainfo_len = EIS_DATAINFO_LEN; 
	break;
      
      case SB_APID_FPPE_UDUMP:
      default: datainfo_len = 0; 
      }

      /* == Sequence Flag Check == */
      if( !_stream_close0[id] ){ 
	/* ...-<... */
	EvsSet( MODULE "DISCARD" ); /* no end */
	errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	errlog( MODULE " discarded a stream (force begin)\n" );
	EvsSet( MODULE "ERR:FLAG1" );
	errlog( MODULE " unexpected start flag at count=%d APID=%03x\n", count, apid);
      }else{               
	/* ...><... */
	/* not error */
      }

      _stream_error[id] = false;

      do /* read mdp packet header with checking */ {
	int expected_length;

	/* == Sequence Count Check == */
	/* Any Sequence Counter is OK for new stream */
      
	/* == Squence Flag Check == */
	if( stream_close ){
	  EvsSet( MODULE "ERR:LENGTH3" ); /* unexpected flag */
	  errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	  errlog( MODULE " unexpected sequence flag (11) detected\n" );
	  _stream_error[id] = true;
	  break;
	}

	/* == Packet Size Check == */
	expected_length = SBccsdsPacketCSize  
	  + verinf_len + mdphdr_len + datainfo_len; 

	if( length != expected_length ){
	  EvsSet( MODULE "ERR:LENGTH1" ); /* header size error */
	  errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	  errlog( MODULE " unexpected size (%d) for first packet\n", length);
	  errlog( MODULE " expected size is (%d) for first packet\n", 
		  expected_length );
	  _stream_error[id] = true;
	  break;
	}
      
	/* .. processing .. */
	byte const PTR packet = _bi.getPacket();

	for(int i=0; i<length; i++)
	  _packet[id][i] = packet[i];

	_wrpntr[id] = 0;

      } while( ALWAYS_ONLY_ONCE );

      int packet_len = SBccsdsPacketCSize;

      for(int i=0; i<verinf_len;   i++)
	_bo.setVerInf  (id, i, _packet[id][packet_len+i]);
      packet_len += verinf_len;
      
      for(int i=0; i<mdphdr_len;   i++)
	_bo.setMdpHdr  (id, i, _packet[id][packet_len+i]);
      packet_len += mdphdr_len;
      
      for(int i=0; i<datainfo_len; i++)
	_bo.setDataInfo(id, i, _packet[id][packet_len+i]);
      packet_len += datainfo_len;
      
      _bo.setVerInfLen  ( id, verinf_len ); 
      _bo.setMdpHdrLen  ( id, mdphdr_len ); 
      _bo.setDataInfoLen( id, datainfo_len ); 

    }else{

      do /* read mdp data part with aboring by checking */ {
	int len;

	if( _stream_error[id] ) break;

	/* == Sequence Flag Check == */
	if( _stream_close0[id] ){
	  /* ...>-... */
	  EvsSet( MODULE "ERR:FLAG2" );
	  errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	  errlog( MODULE " start flag missing at count=%d APID=%03x\n", 
		  count, apid);
	  _stream_error[id] = true;
	  break;
	}else{
	  /* ...--... */
	  /* no error */
	}

	/* == Sequence Counter Check == */
	if( count != COUNTUP(_packet_count0[id]) ){
	  EvsSet( MODULE "ERR:COUNT" );
	  errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	  errlog( MODULE " discard a stream with unexpected counter change\n");
	  _stream_error[id] = true;
	  break;
	}

	/* == Packet Size Check == */
	if( !stream_close ){
	  int expected_length = 2042;
	  if( length != expected_length ){
	    EvsSet( MODULE "ERR:LENGTH2" ); /* header size error */
	    errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	    errlog( MODULE " unexpected packet size (%d) for continuing image data\n", length );
	    ////	    _stream_error[id] = true;
	    break;
	  }
	}else{
	  /* impossible to detect error */
	}

	/* .. processing .. */
	int packet_len = SBccsdsPacketCSize;
	byte const PTR packet = _bi.getPacket();

	len = length - SBccsdsPacketCSize;

	for(int i=0; i<len; i++)
	  _imgbuf[id][_wrpntr[id]+i] = packet[packet_len+i];

	packet_len += len;

	_wrpntr[id] += len;
      } while( ALWAYS_ONLY_ONCE );
    }

    _stream_close0[id] = stream_close;
    _packet_count0[id] = count;

    if( !_stream_close0[id] ){
      /* continuation of reading a stream */ 

      status = ANL_SKIP;
    }else if( _stream_error[id] ){
      /* discard whole stream at the end */ 
      EvsSet( MODULE "DISCARD" ); /* error detect type */
      errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
      errlog( MODULE " discarded a stream (at the end)\n");
      status = ANL_SKIP;
    }

    if( status != ANL_SKIP ){
      ///      printf("DePacketize: %06x\n", _wrpntr[id] );

      int packet_len = SBccsdsPacketCSize;

      for(int i=0; i<_wrpntr[id]; i++){
	if( i >= CMPIMG_LEN ){
	  EvsSet( MODULE "ERR:TOOLARGE" ); /* error detect type */
	  errlog( MODULE " [TI=%08x, sbtime=%15.4f] APID=%03x\n", ti, sbtime, apid );
	  errlog( MODULE " too large image \n");
	  break;
	}
	_bo.setCmpImg( i, _imgbuf[id][i]);
      }

      _bo.setCmpImgLen  ( _wrpntr[id] );
      _bo.bnkput(id);
    }
  }
};

/*
 * Linkage for slrb_anl
 */

extern "C" {
  char DePacketize_version[] = "version 20050920";

  static class DePacketize lc = NEW DePacketize();

  void DePacketize_init(int *status){ 
    lc.init();
    *status = ANL_OK;
  }
  void DePacketize_ana(int nevent, int eventid, int *status){ 
    lc.ana( *status ); 
  }
  
  /* default functions */
  
  void DePacketize_startup(int *status){ *status = ANL_OK; }
  void DePacketize_com    (int *status){ *status = ANL_OK; }
  void DePacketize_his    (int *status){ *status = ANL_OK; }
  void DePacketize_bgnrun (int *status){ *status = ANL_OK; }
  void DePacketize_endrun (int *status){ *status = ANL_OK; }
  void DePacketize_exit   (int *status){ *status = ANL_OK; }
}


