;+
; NAME: eis_md_decomp.pro
;
; PURPOSE: Decompress EIS mission data packets.
;      
; CATEGORY: Science
;
; CALLING SEQUENCE: eis_mk_decomp, 'md_archive'
;
; INPUTS: Filename string of the mission data archive
;
; KEYWORD PARAMETERS: None
;
; OUTPUTS: One decompressed mission data archive in /tmp
;
; CALLS: Various SSW fits routines.
;
; COMMON BLOCKS:None.
;
; PROCEDURE:
;
; RESTRICTIONS: Uses SolarSoft
;
; MODIFICATION HISTORY:
;	23/11/05 mcrw	wrote
;       14/08/06 mcrw	added documentation
;
;-
pro eis_md_decompress::remove_temporary_files
    print, 'Removing temporary files'
    cmd = '/bin/rm ' + getenv('HOME') + '/work/tmp/eis_md*'
    spawn, cmd
end

pro eis_md_decompress::clear_up, flag=flag

    if flag eq 'FAIL' then begin
        if self.merge then begin
            self.destination = getenv('HOME') + '/work/localdata/sdtp/merge/decompressed/'
        endif else begin
            self.destination = getenv('HOME') + '/work/localdata/sdtp/decompressed/'
        endelse
        cmd_line = strarr(3)
        cmd_line[0] = 'cp'
        cmd_line[1] = filename
        cmd_line[2] = self.destination
        spawn, cmd_line, result, /noshell
        self->close_record_file
        obj_destroy, md_archive
end

pro eis_md_decompress::open_record_file
    home = getenv('HOME')
    if keyword_set(merge) then begin
        path = '/work/localdata/log/decompression/merge_decomp_record.txt'
    endif else begin
        path = '/work/localdata/sdtp/decompressed/decomp_record.txt'
    endelse
    file = home + path
    openw, lu, file, /get_lun, /append
    self.record_file = file
    self.record_unit = lu
end

function eis_md_decompress::good_flags, flags
    flags_ok = 0
    case flags of
        '11' : flags_ok = 1
        '13' : flags_ok = 1
        '31' : flags_ok = 1
        '33' : flags_ok = 1
        '23' : flags_ok = 1     ; should really be bad flags
        else: print, 'Bad flags: ', flags ; May remove
    endcase
    return, flags_ok
end

pro eis_md_decomress::set_compression, compression
    if ((compression and 'FF00'x) or '3B00'x) eq '3B00'x then begin
        self.decomp_program = progs + '/dpcmtopnm ' + temp_file + ' > ' + temp_file + '_decomp'
        print, 'DPCM compression'
        self.comp_type = 'DPCM'
    endif else begin
        if ((compression and 'FF00'x) or '3F00'x) eq '3F00'x then begin
            self.decomp_program = progs + '/djpeg ' + temp_file + ' > ' + temp_file + '_decomp'
            print, 'JPEG compression'
            self.comp_type = 'JPEG'
            index = compression - '3F28'X
            if index lt 8 then self.comp_type = self.ccomp_type + '_' + self.comp_factors[index]
        endif
    endelse
    self.bit_compressed = (compression and '7800'x) ne 0
;        if(compression and '7800'x) ne 0 then begin
;            print, 'Doing bit decompression...'
;            bit_data = eis_bit_decomp(decompressed_image)
;        endif else begin
;            print, 'NO BIT COMPRESSION...'
;            bit_data = decompressed_image
;        endelse
end

pro eis_md_decompress::rebuild_archive, bit_data
    byteorder, bit_data, /swap_if_little_endian

    if self.merge) then begin
        out_file = getenv('HOME') + '/work/localdata/sdtp/merge/decompressed/' + basename
    endif else begin
        out_file = getenv('HOME') + '/work/localdata/sdtp/decompressed/' + basename
    endelse
    openw, wlu, out_file, /get_lun, /append
    self->eis_write_decompressed_packet, rlu, wlu, archive->first_ccsds_header(), archive->md_header(), bit_data
    close, wlu
    free_lun, wlu
end

function eis_md_decompress::perform_bit_decompression, decompressed_image
    if self.bit_compressed then begin
        print, 'Doing bit decompression...'
        bit_data = eis_bit_decomp(decompressed_image)
    endif else begin
        print, 'NO BIT COMPRESSION...'
        bit_data = decompressed_image
    endelse
    return, bit_data
end

pro eis_md_decompress::remove_bad_image_file
    cmd_line[0] = 'rm'
    cmd_line[1] = temp_file + '_decomp'
    cmd_line[2] = ''
    spawn, cmd_line, result, /noshell
end

pro eis_md_decompress::write_temporary_data, data
    ; Now write the compressed data out
    ; Create temporary filename
    openw, lu, self.temp_image_file, /get_lun
    writeu, lu, data
    close, lu
    free_lun, lu
end

function eis_md_decompress::read_decompressed_image, decompressed_image
    print, 'Reading decompressed image'
    ok = 0
    openr, brl, self.temp_image_file + '_decomp', /get_lun
    ;
    ; Check decompressed file size is not zero
    ;
    res = fstat(brl)
    if res.size ne 0 then begin
        decompressed_image=read_binary(brl,data_type=2)
        ok = 1
    endif else begin
        print, self.basename + ' failed decompression'
        if self.merge then begin
            dest = getenv('HOME') + '/work/localdata/sdtp/merge/bad_decompress/'
            cmd_line = strarr(3)
            cmd_line[0] = 'mv'
            cmd_line[1] = getenv('HOME') + '/work/localdata/sdtp/merge/' + 'basename'
            cmd_line[2] = dest
            spawn, cmd_line, result, /noshell
            self->remove_bad_image_file
            goto, bad_archive
        endif
    endelse
    close, brl
    free_lun, brl
    return, ok
end

; Create a single mission data packet from the decompressed data
; rl is the read lun
; wl is the write lun
; ccsdsHeader is the header of the first ccsds packet in the original archive
; mdHeader is the first md header of the original archive
; buffer is an intarr of pixels
pro eis_md_decompress::eis_write_decompressed_packet, rl, wl, ccsds_header, md_header, buffer
;    on_ioerror, badRd

    total_pixels = long(n_elements(buffer))
print,'Writing ', total_pixels, ' decompressed pixels out'
    pixel_pointer = 0L
    pixels_left = total_pixels - pixel_pointer

    hdr = ccsds_header
    hdr[2] = hdr[2] and 2^6 - 1 ; Clear flag to make sure or works ok
    seq_count = ishft(hdr[2], -6) or hdr[3]
    seq_flag = 0

    ; Set sequence flag
    if pixels_left lt 1016 then seq_flag = 3

    ; Write out md header
    writeu, wl, ccsds_header
    writeu, wl, md_header

    ; Now read the decompressed data and write back out in ccsds packets
    going = 1
    while going do begin			; Try to write a full pixel ccsds packet
        ; Create ccsds packet with the data
        seq_count = seq_count + 1
        if pixels_left gt 1016 then begin ; Full packet
            hdr[2] = hdr[2] or ishft(seq_count, -8)
            hdr[3] = (seq_count and 255)
            hdr[4] = 7          			; MSB length
            hdr[5] = 243				; LSB length
            writeu, wl, hdr
            writeu, wl, buffer[pixel_pointer:(pixel_pointer+1015)]
            pixel_pointer = pixel_pointer + 1016
            pixels_left = total_pixels - pixel_pointer
            going = pixels_left gt 1016
        endif
    endwhile

    ; Packet up residual data
    if pixels_left gt 0 then begin
        bytes_left = pixels_left * 2
        seq_count = seq_count + 1
        seq_flag = 2
        hdr[2] = hdr[2] or ishft(seq_flag, 6)		; Sequence flag
        hdr[2] = hdr[2] or ishft(seq_count, -8)
        hdr[3] = (seq_count and 255)
        bytes_left = bytes_left + 3
        hdr[4] = ishft((bytes_left and 65280), -8)	; MSB length
        hdr[5] = bytes_left and 255			; LSB length
        writeu, wl, hdr
        writeu, wl, buffer[pixel_pointer:*], transfer_count=t_count
    endif
end

; Process a single md packet
; archive is the ccsds reader object
; compression is the compression number from the mission data header
; basename is the original archive without the path
;function eis_process_packet, archive, compression, basename, integers_in_count, integers_out_count, ctab, ctype, first_time, merge=merge
function eis_md_decompress::process_packet, archive, compression, basename, integers_in_count, integers_out_count, ctab, ctype, first_time, merge=merge

    ; Path to the decompression programs
    progs = getenv('EIS_BIN')

    ; Get the 8 byte table in the header
    hdr = archive->md_header();
    comp_table = hdr[0:7]
    ctab = comp_table

    ; Read in the rest of archive
    if archive->read_ccsds_header() then begin
        data = archive->read_user_data()

        ; data is a BYTE array        
        going = 1
        while going do begin
            if archive->is_end_packet_of_sequence() then begin
                going = 0
            endif else begin
                foo = archive->read_ccsds_header()
                if foo eq 0 then going = 0 else data = [data, archive->read_user_data()]
            endelse
        endwhile 
;
; Apply fix for mdp bug - mcrw 24 Jan 07
;
        mdppmdctrecover_merge, hdr, data, first_time
        self.integers_in_count = self.integers_in_count + (n_elements(data) / 2)

        ; Now write the compressed data out
        self->write_temporary_data, data

        ; Now do image decompression
        ; 0x3B28 : DPCM
        ; 0x3F28 : JPEG        
        if self.decomp_program eq '' then return, 0

        print, 'Doing ' + self.comp_type + ' image decompression...'

        ; Spawn a child process to do the image decompressing
        print, self.decomp_program
        spawn, self.decomp_program

        ; Now read the decompressed image data
        ; and perform bit compression on it
        ok = self->read_decompressed_image(decompressed_image)
        if not ok then goto, bad_archive

        ; Update integer decompressed count
        print,'Read ', n_elements(decompressed_image), ' decompressed pixels'
        self.integers_out_count = self.integers_out_count + n_elements(decompressed_image)

        ; Do bit decompression and write bit decompressed data to temporary file
        bit_data = self->perform_bit_decompression(decompressed_image)

        ; Now rebuild the archive
        self->rebuild_archive, bit_data

        return, 1
    endif

    ; No packets left in archive
    ; Premature end
bad_archive:
    return, 0
end

pro eis_md_decompress::initialize, filename, merge=merge
    self.filename = filename
    p = strpos(filename, '/', /reverse_search)
    self.basename = strmid(filename, p + 1, 255)
    self.integers_in_count = 0L
    self.integers_out_coubnt = 0L
    self.comp_factors = ['98', '90', '75', '50', '95', '92', '85', '65']
    self.temp_file = getenv('HOME') + '/work/tmp/' + 
    if keyword_set(merge) then begin
        self.merge = 1
        self.out_file = getenv('HOME') + '/work/localdata/sdtp/merge/decompressed/' + self.basename
    endif else begin
        self.out_file = getenv('HOME') + '/work/localdata/sdtp/decompressed/' + self.basename
    endelse
    self.temp_image_file = getenv('HOME') + '/work/tmp/' + self.basename + '_image'
end

;pro eis_md_decompress::open_record_file, file
;    openw, lu, file, /get_lun, /append
;    self.record_file = file
;    self.record_unit = lu
;end

pro eis_md_decompress::close_record_file
    lu = self.record_unit
    close, lu
    free_lun, lu
end

;
pro eis_md_decompress::report
    print, 'Processed: ', processed
    print, 'Integers in : ', integers_in_count
    print, 'Integers out: ', integers_out_count
    print, 'Compression: ', (float(integers_in_count)/float(integers_out_count))*100.0
		;        printf,rat,'Comp table ',ctab,' type ',ctype, string(compression,format='(Z04)')
		;        printf,rat,'Integers in : ',integers_in_count
		;        printf,rat,'Integers out: ',integers_out_count
		;        printf,rat,'Compression: ', (float(integers_in_count)/float(integers_out_count))*100.0
		;        printf,rat,''
		;;        dint_in  = double(integers_in_count)
		;;        dint_out = double(integers_out_count)
		;;        comp_amount = (dint_in / dint_out) * 100.0
		;;        printf, rat, format='(A26, 2(I15), 2(D15.2), A10)', $
		;;          filename, $
		;;          integers_in_count, $
		;;          integers_out_count, $
		;;          (double(integers_in_count) / double(integers_out_count)) * 100.0, $
		;;          double(integers_out_count) / double(integers_in_count), $
		;;          ctype
    printf, rat, format='(A26, 2(I15), 2(D10.2), Z6, A10, "  [",8(Z02),"]")', $
      newfilename, $
      integers_in_count, $
      integers_out_count, $
      (double(integers_in_count) / double(integers_out_count)) * 100.0, $
      double(integers_out_count) / double(integers_in_count), $
      compression, $
      ctype, $
      ctab
end

; Main procedure.
pro eis_md_decompress::eis_md_decomp_fix2, filename, merge=merge
    self->open_record_file, merge=merge
    self->initialize, filename, merge=merge

    self.md_archive = obj_new('eis_ccsds_interface')
    self.md_archive->open_for_reading, filename, ok
    if not ok then begin
        print, 'No such file'
        self->clear_up
        obj_destroy, self.md_archive
        return
    endif

    ; Read in decompression factors (a, b, c, Nc)

    ; Get the first md header
    self.md_archive->read_md_header_packet()
    if not self.md_archive->is_compressed() then begin
        print, 'Archive not compressed'
        ; Close files
        self.md_archive->close_input
        ; Move stuff about
        self->clear_up
        return
    endif

    compression = self.md_archive->get_compression_method()
    self->set_compression, compression
    flags = self.md_archive->get_md_sequence_flags()
    if self->good_flags(flags) then begin
        print, 'Processing for flags: ', flags
;            processed = eis_process_packet(md_archive, compression, newfilename, integers_in_count, integers_out_count, ctab, ctype, first_time_round, merge=merge)
        processed = self->process_packet(merge=merge)
        if processed then begin
            self->remove_temporary_files

            self->report;

;            print, 'Processed: ', processed
;            print, 'Integers in : ', integers_in_count
;            print, 'Integers out: ', integers_out_count
;            print, 'Compression: ', (float(integers_in_count)/float(integers_out_count))*100.0
;		;        printf,rat,'Comp table ',ctab,' type ',ctype, string(compression,format='(Z04)')
;		;        printf,rat,'Integers in : ',integers_in_count
;		;        printf,rat,'Integers out: ',integers_out_count
;		;        printf,rat,'Compression: ', (float(integers_in_count)/float(integers_out_count))*100.0
;		;        printf,rat,''
;		;;        dint_in  = double(integers_in_count)
;		;;        dint_out = double(integers_out_count)
;		;;        comp_amount = (dint_in / dint_out) * 100.0
;		;;        printf, rat, format='(A26, 2(I15), 2(D15.2), A10)', $
;		;;          filename, $
;		;;          integers_in_count, $
;		;;          integers_out_count, $
;		;;          (double(integers_in_count) / double(integers_out_count)) * 100.0, $
;		;;          double(integers_out_count) / double(integers_in_count), $
;		;;          ctype
;            printf, rat, format='(A26, 2(I15), 2(D10.2), Z6, A10, "  [",8(Z02),"]")', $
;              newfilename, $
;              integers_in_count, $
;              integers_out_count, $
;              (double(integers_in_count) / double(integers_out_count)) * 100.0, $
;              double(integers_out_count) / double(integers_in_count), $
;              compression, $
;              ctype, $
;              ctab

        endif else printf, rat, format='(A26,I5)', newfilename, processed

        close,rat
        free_lun,rat

    endif else print, 'Flags indicate this is not the start of a raster' ; Will log this to a file
            
            ; Close files
    self->close_record_file
    md_archive->close_input
    obj_destroy,md_archive

end

pro eis_md_decompress__define
    struct = { eis_md_decompress,              $
               filename           :        '', $
               merge              :         0, $
               record_file        :        '', $
               record_unit        :        98, $
               rat                :        99, $
               basename           :        '', $
               integers_in_count  :        0L, $
               integers_out_count :        0L, $
               md_archive         : obj_new(), $
               out_file           :        '', $
               temp_image_file    :        '', $
               destination        :        '', $
               command_line       :        '', $
               decomp_program     :        '', $
               comp_type          :        '', $
               bit_compressed     :         0, $
               comp_factors       :  strarr(8)
               
    }
end
