;+
; NAME:
;       EIS_MK_FITSNAME
;
; PURPOSE:
;	Based on data level (0, 1, 2) and time in UTC (CCSDS format - 
;       YYYY-MM-DDTHH:MM:SS.SSS - see GET_UTC.PRO in SolarSoft), 
;       construct FITS file name in right for EIS on Solar-B. 
;
;
; CATEGORY:
;       Hansteen/Wikstol data processing software
;
; CALLING SEQUENCE:
;       filename = eis_mk_fitsname(fitslevel, utc)
;
; INPUTS:
;       level: Level of data to be written to FITS file (0, 1, 2)
;       data : EIS data object
;	hdr: EIS MD hdr object
;
; KEYWORD PARAMETERS:
;       None
;       
;
; OUTPUTS:
;       Name of FITS file in the format of 'eis_lx_yyyymmdd_hhmmss.fits'
;
; CALLS:
;
;
; COMMON BLOCKS:
;
;
; PROCEDURE:
;
; RESTRICTIONS:
;
;
; MODIFICATION HISTORY:
;       Version 1: 26-Jan-2005: Oivind Wikstol
;        August 2006 - Oivind Wikstol: Re-written to fit new FITS naming convention
;        September 2006 - Oivind Wikstol: Re-written to do time
;                         conversion
;        13 Nov 2006 - ViggoH/MattW: Converted dhu to mdp time, added
;                      yrhr_str keyword
;-
function eis_mk_fitsname, level, data, hdr, yrhr_str=yrhr_str

; get filename of mission data (CCSDS) file:
  ccsds_file = data->getfilename()

; remove directory from filename
  md_file = fname_extract(ccsds_file)

; if in correct format, MD filename should now be in the format:
; eis_md_yyyymmdd_hhmmssttll (t = tenths of a sec, ll is millisec)

; extract yyyymmdd:
  yr_str = strmid(md_file, 7, 8)
; extract hhmmss:
  hr_str = strmid(md_file, 16, 6)

  !quiet = 0
  message, 'ccsds_file = ' + string(ccsds_file), /info
  message, 'md_file = ' + string(md_file), /info
  message, 'yr_str = ' + string(yr_str), /info
  message, 'hr_str = ' + string(hr_str), /info
  !quiet = 1

; get shutter open time of exposure 0:
   i=0
   repeat begin
     ti0 = ((hdr[i]-> getexp_info()).ti_1)
     i=i+1
   endrep until ti0 ne 0
   ti0 = ti0/16 ; convert from dhu to mdp time 
   top_bits=ulong((data->getccsds_packet_time()).first)
   ti0 = string((top_bits/ulong(2)^28)*ulong(2)^28+ti0)

; do time conversion:
  yrhr_str = yr_str+hr_str
  ut_str = eis_ti2utc(string(ti0), yrhr_str)

; if successful, ut_str is now 'yyyymmddThhmmss' or yyyymmdd_hhmmss for failed conversion
  converted_ok = strpos(ut_str,'T',8)

  ; mcrw 20221015
  ;if converted_ok eq -1 then ut_str = '99999999_999999'
  if converted_ok eq -1 then ut_str = yr_str + '_' + hr_str

; get rid of 'T' in lieu of '_'
  strput,ut_str,'_',8  
  
; put it all together
  filename = 'eis_' + level + '_' + ut_str + '.fits'

  ; mcrw 20221015
  ; EIS_MK_FITSNAME: filename = eis_l0_99999999_999999.fits
  ; EIS_MK_FITSNAME: filename = eis_l0_20220927_053928.fits
  !quiet = 0
  message, 'filename = ' + filename, /info
  !quiet = 1

  return, filename
end
