function eis_ti2utc, ti, grt
;+
; NAME: eis_ti2utc.pro
;
; PURPOSE: Converts spacecraft time to utc
;
; CATEGORY: Science
;
; CALLING SEQUENCE: result = eis_ti2utc(ti, grt)
;                   result = eis_ti2utc('0033D637',' 20050811113116')
;
; INPUTS: ti  : (string) onboard spacecraft time
;         grt : (string) ground received time
;
; KEYWORD PARAMETERS: None
;
; OUTPUTS: If no errors occur in processing the output is a string in
;          in the approved EIS format (yyyymmddThhmmss). In the event
;          of an error the grt is returned, which is the EIS time
;          format without the T.
;
; CALLS: sbtime, the J-side provided c program which does the actual
;        conversion.
;        doy2ex, anytim2cal
;
; COMMON BLOCKS: None.
;
; PROCEDURE: 2 auxilliary files are required for the time conversion,
;            supplied by J-side. One is a CNV_FILE which contains
;            information about time conversion for a specified range
;            in every year and the other is a leap seconds file.
;
; RESTRICTIONS: Uses SolarSoft
;
; MODIFICATION HISTORY:
;       25/08/06 Viggo Hansteen, mcrw	wrote
;       26/08/06 mcrw			added documentation
;					renamed function
;					used environment variable in spawn
;                                       not using strput as gave wrong result
;       26/09/06 mcrw Tries all CNV files
;                     Requires $EIS_TI_BASE
;-

    
;;;print, 'eis_ti2utc inputs: ', ti, ' (', string(ti, format='(Z08)'), ') ', grt
    bin = getenv('EIS_BIN')
    cmd_line = strarr(3)
    cmd_line[0] = bin + '/sbtime'
;;;    cmd_line[1] = ti
    cmd_line[1] = string(ti, format='(Z08)')
    cmd_line[2] = grt

    converted = 0

    date_str = strmid(grt, 0, 8)
    time_str = strmid(grt, 8, 6)        

    ; Requires logic to access correct CNV_FILE - brute force, tries
    ; each CNV file now

    ; Get hold of the conversion files
    ti_base = getenv('EIS_TI_BASE')
    setenv, 'LEAP_FILE=' + ti_base + '/Lepasec.dat'

    files = findfile(ti_base + '/CNV_FILE*', count=count)
    if count eq 0 then begin
        message, 'No date files', /info
        return, date_str + '_' + time_str
    endif

    ; Try each CNV file in turn
    for i = 0, count - 1 do begin
;;;        print,'Trying CNV file ' + files[i]
        setenv, 'BASE_FILE=' + files[i]

        result = ''

        ; Pass the input parameters to the c program. This returns
        ; an array of strings
        spawn, cmd_line, result, /noshell

;    Debug.
;    for i=0,n_elements(result)-1 do begin 
;       print,result[i]
;    endfor

        ; Check the c program could convert the time
        if n_elements(result) gt 4 then begin 
            converted = 1
            break
        endif
    endfor

    if converted eq 0 then begin
        message, 'No date found!! Using ground received time', /info
        return, date_str + '_' + time_str
    endif

    ; Time was converted ok. Get the output string and massage into
    ; the required format

    ; The last string contains the converted time
    result = result[n_elements(result) - 1]

    ; Extract part we want
    result = strmid(result,strpos(result,'TIME')+5,18)

    ; Break into various components
    year = strmid(result,0,4)
    doy = strmid(result,6,3)
    tod = strmid(result,10,8)

    ; Use day number to get date
    date = doy2ex(doy,year,/string)

    ; Reformat
    result = anytim2cal(date+' '+ tod, form = 8)

    ; Insert the T in the correct place
    ; (strput gives incorrect output)
;    strput,result,'T',8
;    return, result
    date_str = strmid(result, 0, 8)
    time_str = strmid(result, 8, 6)
    return, date_str + 'T' + time_str

end
