;+
; NAME:
;       EIS_FITS_DATAID
;
; PURPOSE:
;	Adds keywords related to data identification (Data ID) 
;	to the primary header of the EIS FITS file.
;
;
; CATEGORY:
;       Data Processing Software
;
; CALLING SEQUENCE:
;       eis_fits_dataid, header
;
; INPUTS:
;       header: The primary header which will have keywords added to it.
;	md_hdr: EIS Mission Data header object array
;       plan: EIS planning object
;
; KEYWORD PARAMETERS:
;       none
;
;
; OUTPUTS:
;       The extended primary header
;
; CALLS:
;	fxhread
;	fxaddpar
;
; COMMON BLOCKS:
;
;
; PROCEDURE:
;
; RESTRICTIONS:
;
;
; MODIFICATION HISTORY:
;       xx-Nov-2002: Oivind Wikstol.
;       22-Nov-2006: Viggo Hansteen - Added 'SLIT_IND' tag.
;       23-Apr-2008: Viggo Hansteen - Fixed slit_ind bug.
;        9-May-2008: A. Gardini     - Fixed slint_ind typo.
;       10-Sep-2008: Viggo Hansteen - Set up for new slit position defintions given by Khalid
;                                     Al-Janabi to work from 24-Aug-2008.
;        5-Feb-2019: M. Whillock    - Added sequence_id as parameter
;                                     to function
;-
pro eis_fits_dataid, header, sequence_id, md_hdr, data, plan, datatype
;
; exposure info from mission data header:
  exp_info = md_hdr[0]-> getexp_info()
  slit_nr = exp_info.slit_nr

;  print,'slit_nr',slit_nr
;  print, 'sequence_id', sequence_id

  !quiet = 0
  message, 'slit_nr: ' + strtrim(string(slit_nr), 2), /info
  message, 'sequence_id: ' + strtrim(string(sequence_id), 2), /info
  !quiet = 1

  slit_ind=data->getslit_ind(slit_nr)
  data->setslit_ind,slit_ind
  slit_id=data->getslit_id(slit_ind)
  nexp = exp_info.nexp
  nexp_prp = exp_info.exp_prp
  nraster = 1
  if exp_info.fmir_step gt 0 then nraster = nexp/nexp_prp

  nrppexp = md_hdr[0]->getnr_of_packets() ; number of packets pr. exposure

; get fitslev and fits_reformat info
  get_utc, date, /ccsds  
  home_inst = data-> gethome_inst()
  fits_reformat = data-> getfits_reformat()
  fitslev = data-> getfitslev()

  if fitslev eq 0 then begin
    data-> setfits_reformat, 'date_rf0', date
    data-> setfits_reformat, 'orig_rf0', home_inst
  endif

  fits_reformat = data-> getfits_reformat()

  fxaddpar, header, 'origin', (data-> gethome_inst()),$
                    'Institute where FITS file was written'
  fxaddpar, header, 'data_lev', (data-> getfitslev()), $
                    'FITS Level (0, 1, 2)'
  fxaddpar, header, 'date_rf0', fits_reformat.date_rf0, $
                    'Date and time of Level 0 reformat'
  fxaddpar, header, 'orig_rf0', fits_reformat.orig_rf0, $
                    'Institution where Level 0 reformat was done'
  fxaddpar, header, 'ver_rf0', fits_reformat.ver_rf0, $
                    'Fits Level 0 reformatter version no.'
  fxaddpar, header, 'date_rf1', fits_reformat.date_rf1, $
                    'Date and time of Level 1 reformat'
  fxaddpar, header, 'orig_rf1', fits_reformat.orig_rf1, $
                    'Institution where Level 1 reformat was done'
  fxaddpar, header, 'ver_rf1', fits_reformat.ver_rf1, $
                    'Fits Level 1 reformatter version no.'
  fxaddpar, header, 'telescop', plan-> get_telescop(), $
                    'Telescope (Solar-B)'
  fxaddpar, header, 'instrume', plan-> get_instrume(), $
                    'EUV Imaging Spectrometer'
  fxaddpar, header, 'obstitle', plan-> get_obstitle(), $
                    'Title of the observation'
  fxaddpar, header, 'target', plan-> get_target(), $
                    'Observation region (QS, CH, AR, FS)'
  fxaddpar, header, 'noaa_num', plan-> get_noaa_num(), $
                    'Active region NOAA number'
  fxaddpar, header, 'sci_obj', plan-> get_sci_obj(), $
                    'Target phenomena for this observation'
  fxaddpar, header, 'obs_dec', plan-> get_obs_dec(), $
                    'Properties of the observation'
  fxaddpar, header, 'join_sb', plan-> get_join_sb(), $
                    'Joint observation indicator'
  fxaddpar, header, 'obs_num', plan-> get_obs_id(), $
                    'Sequential number of Solar-B observation'
  fxaddpar, header, 'jop_id', plan-> get_jop_id(), $
                    'Joint observing program number'
  fxaddpar, header, 'rast_acr', plan-> get_rast_acr(), $
                    'Raster Acronym'
  fxaddpar, header, 'stud_acr', plan-> get_study_acr(), $
                    'Study Acronym'
  fxaddpar, header, 'll_acr', plan-> get_ll_acr(), $
                    'Line List Acronym'
  fxaddpar, header, 'll_id', plan-> get_ll_id(), $
                    'Line List ID'
  fxaddpar, header, 'study_id', plan-> get_study_id(), $
                    'Study ID'
  fxaddpar, header, 'st_auth', plan-> get_study_auth(), $
                    'Study Author'
  fxaddpar, header, 'observer', plan-> get_observer(), $
                    'Chief observer name'
  fxaddpar, header, 'planner', plan-> get_planner(), $
                    'Name of Chief Planner'
  fxaddpar, header, 'tohbans', plan-> get_tohban(), $
                    'Name of RT Tohbans'
  fxaddpar, header, 'datatype', plan-> get_datatype(), $
                    'Type of data (pixel data or moments)'
  fxaddpar, header, 'timesys', 'UTC', $
                    'Time system used in FITS file'
  fxaddpar, header, 'main_id', md_hdr[0]-> getmain_id(), $
                    'Main ID'
  fxaddpar, header, 'mdpack', fix(md_hdr[0]->getnr_of_packets()), $
                    'Number of telemetry packets per exposure'
  fxaddpar, header, 'seq_nr', fix(exp_info.seq_nr), $
                    'Sequence table number' 
  fxaddpar, header, 'slit_nr', exp_info.slit_nr, $
                    'Slit number'
  fxaddpar, header, 'slit_ind', slit_ind, $
                    'Slit index'
  fxaddpar, header, 'slit_id', slit_id, $
                    'Slit ID'
  fxaddpar, header, 'll_nr', fix(exp_info.ll_nr), $
                    'Line list number'
  fxaddpar, header, 'nwin', fix(exp_info.nwin), $
                    'Number of line windows'

;;; 20190205 mcrw
;;;  fxaddpar, header, 'tl_id', exp_info.seq_id, $
  fxaddpar, header, 'rec_tlid', exp_info.seq_id, 'Received timeline entry id'
            
  fxaddpar, header, 'tl_id', sequence_id, $
                    'Unique number for each timeline entry'  

  fxaddpar, header, 'tl_wrap', (sequence_id gt 65535) ? 'True' : 'False', $
                    'True if the on-board timeline id has wrapped'

  fxaddpar, header, 'plan_tl', plan->get_tl_id(), $
                    'Timeline id entry generated by the PT'  

  fxaddpar, header, 'rast_id', exp_info.rast_id, $
                    'Unique number for each raster' 
  fxaddpar, header, 'fmir_ss', exp_info.fmir_step, $
                    'Fine mirror step size (0 for sit-and-stare mode)'
  fxaddpar, header, 'nexp', nexp, $
                    'Number of exposures performed in raster'
  fxaddpar, header, 'rast_req', exp_info.nexp, $
                    'Number of raster exposures required'
  fxaddpar, header, 'nexp_prp',fix(nexp_prp) , $
                  'Number of exposures pr raster position'
  fxaddpar, header, 'nraster', nraster, $
                    'Number of unique raster positions'
  fxaddpar, header, 'ccdxl', exp_info.ccd_length, $
                    'CCD X-length incl. scan pix' 
  fxaddpar, header, 'xws', exp_info.xws, $
                    'HW (Cam) x window start'
  fxaddpar, header, 'xw', exp_info.xw, $
                    'HW (Cam) x window length'
  fxaddpar, header, 'yws', exp_info.yws, $
                    'HW (Cam) y window start'
  fxaddpar, header, 'yw', exp_info.yw, $
                    'HW (Cam) y window height'
  fxaddpar, header, 'rrep_req', exp_info.nrepr, $
                    'Number of raster repeats required'
  fxaddpar, header, 'rrep_pfd', exp_info.nrepp, $
                    'Number of raster repeats performed'
  fxaddpar, header, 'srep_req', fix(exp_info.nseqr), $
                    'Number of sequence repeats required'
  fxaddpar, header, 'srep_pfd', fix(exp_info.nseqp), $
                    'Number of sequence repeats performed'
end
