;+
; NAME:
;       EIS_FITS_OBSTIME
;
; PURPOSE:
;	Adds keywords related to observation time 
;	to the primary header of the EIS FITS file.
;
;
; CATEGORY:
;       Data Processing Software
;
; CALLING SEQUENCE:
;       eis_fits_obstime, header
;
; INPUTS:
;       header: The primary header which will have keywords added to it.
;	md_hdr: EIS Mission Data header object array
;       plan: EIS plannnig object
;
; KEYWORD PARAMETERS:
;       none
;
;
; OUTPUTS:
;       The extended primary header
;
; CALLS:
;	fxhread
;	fxaddpar
;
; COMMON BLOCKS:
;
;
; PROCEDURE:
;
; RESTRICTIONS:
;
;
; MODIFICATION HISTORY:
;       xx-nov-2002  : Oivind Wikstol.
;       xx-aug-2006  : Added exptime
;                      NOTE: DATE_OBS AND DATE_END ARE IN WRONG FORMAT. NO TIME 
;                      CONVERSION BETWEEN TI AND UTC AVAILABLE!!!
;	    19-sep-2006  : OW - Removed exptime
;       13-nov-2006  : VH/MW - convert dhu to mdp time 
;                             added yrhr_str parameter so that
;                             time can be converted - time is now converted correctly
;       16-dec-2006  : VH - added time of first and last ccsds packet
;-
pro eis_fits_obstime, header, md_hdr, plan, yrhr_str, data, ot=ot
;
; when a time conversion routine is available, use these times along with conversion to 
; get right time. for now, use string of space craft times...
; division by 16 to convert from dhu to mdp time
; 
  first_packet_ti=ulong((data->getccsds_packet_time()).first)
  last_packet_ti=ulong((data->getccsds_packet_time()).last) 
  i=0
  repeat begin
    obt_time = ulong((md_hdr[i]-> getexp_info()).ti_1)
    i=i+1
  endrep until obt_time ne 0
  i=n_elements(md_hdr)-1
  repeat begin
    obt_end = ulong((md_hdr[i]-> getexp_info()).ti_2)
    i=i-1
  endrep until obt_end ne 0
  top_bits=first_packet_ti
  start_obs = string((top_bits/ulong(2)^28)*ulong(2)^28+obt_time/16)
  start_ut_str = eis_ti2utc(start_obs, yrhr_str)
  converted_ok = strpos(start_ut_str,'T',8)
  ;print, 'start_obs: ' + start_obs + ', yrhr_str: ' + yrhr_str + ', start_ut_str: ' + start_ut_str + ', converted_ok = ' + strtrim(string(converted_ok),2)
  !quiet = 0
  message, 'start_obs: ' + strtrim(string(start_obs), 2) + ', yrhr_str: ' + strtrim(string(yrhr_str), 2) + ', start_ut_str: ' + strtrim(string(start_ut_str), 2) + ', converted_ok = ' + strtrim(string(converted_ok),2), /info
  !quiet = 1
  if converted_ok eq -1 then start_ut_str = '99999999_999999'
; get rid of 'T' in lieu of '_'
  strput,start_ut_str,'_',8  
; check if rollover occurs between first ccsds packet time and obt_end
  if first_packet_ti gt obt_end/16 then top_bits=last_packet_ti else top_bits=first_packet_ti
  end_obs = string((top_bits/ulong(2)^28)*ulong(2)^28+obt_end/16)
  end_ut_str = eis_ti2utc(end_obs, yrhr_str)
  converted_ok = strpos(end_ut_str,'T',8)
  if converted_ok eq -1 then end_ut_str = '99999999_999999'
; get rid of 'T' in lieu of '_'
  strput,end_ut_str,'_',8  
; now convert to format required by Shimojo document
  start_ut_str=eistim2utc(start_ut_str)
  start_ut_shi=start_ut_str
  end_ut_str=eistim2utc(end_ut_str)
  end_ut_shi=end_ut_str
  ot={o_time,start_obs:start_obs, $
               start_ut_str:start_ut_str, $
               start_ut_shi:start_ut_shi,$
               end_obs:end_obs, $
               end_ut_str:end_ut_str, $
               end_ut_shi:end_ut_shi}
  
;  exptime = plan-> get_exptime()

  fxaddpar, header, 'date_obs', start_ut_shi, $
                    'Start date/time of observation'
  fxaddpar, header, 'date_end', end_ut_shi, $
                    'End date/time of observation'
  fxaddpar, header, 'obt_time', obt_time, $
                    'Spacecraft start time (TI (mdp) units)'
  fxaddpar, header, 'obt_end', obt_end, $
                    'Spacecraft end time (TI (mdp) units)'
;  fxaddpar, header, 'exptime', exptime, $
;                    'Requested exposure time from planning tool'

  ; mcrw 20221015
  fxaddpar, header, 'time_conv', converted_ok, $
                    'Spacecraft time conversion ok'

;               
; now find and convert ccsds packet times for first and last ccsds packets.
;
  first_packet_ti=ulong((data->getccsds_packet_time()).first)
  first_packet_str = string(first_packet_ti)
  first_packet_str = eis_ti2utc(first_packet_str, yrhr_str)
  converted_ok = strpos(first_packet_str,'T',8)
  if converted_ok eq -1 then first_packet_str = '99999999_999999'
  last_packet_ti=ulong((data->getccsds_packet_time()).last)
  last_packet_str = string(last_packet_ti)
  last_packet_str = eis_ti2utc(last_packet_str, yrhr_str)
  converted_ok = strpos(last_packet_str,'T',8)
  if converted_ok eq -1 then last_packet_str = '99999999_999999'
; now convert to format required by Shimojo document
  first_packet_str=eistim2utc(first_packet_str)
  last_packet_str=eistim2utc(last_packet_str)
; write to fits header
  fxaddpar, header, 'ccsds_obs', first_packet_str, $
                    'Time of first CCSDS packet'
  fxaddpar, header, 'ccsds_end', last_packet_str, $
                    'Time of last CCSDS packet'
  fxaddpar, header, 'ccsds_tis', first_packet_ti, $
                    'Time of first CCSDS packet (TI (dhu) units)'
  fxaddpar, header, 'ccsds_tie', last_packet_ti, $
                    'Time of last CCSDS packet (TI (dhu) units)'
  
end
