;+
; NAME:
;       EIS_FITS_SCRIPT
;
; PURPOSE:
;	Makes FITS files out of telemetry packets for the 
;	EIS instrument on Solar-B from a list of CCSDS files.
;   Either you give a file name (including directory tree)
;   or you giv a directory name, in which case a list of 
;   ccsds files will be given, and FITS files will be produced for each 
;   CCSDS file.
;
; CATEGORY:
;       EIS Data Processing Software
;
; CALLING SEQUENCE:
;       eis_fits_script,  in_dir, out_dir, logfile = logfile
; INPUTS:
;       in_dir  : Telemetry file directory
;       out_dir : Location of fits files 
;
; KEYWORD PARAMETERS:
;       logfile : Construct logfile containing list of processed and produced files 
;
; OUTPUTS:
;
; CALLS:
;
;       eis_mkfits       
;
; COMMON BLOCKS:
;
; PROCEDURE:
;
; RESTRICTIONS:
;
; MODIFICATION HISTORY:
;   Oivind Wikstol   12-Sep-2005 - Original version
;   Matthew Whillock 18-Nov-2006 - Rewrite
;   Viggo Hansteen   23-Nov-2006 - Added logging of file names
;   Viggo Hansteen   10-Dec-2006 - Name change from fits_script to eis_fits_script -
;                                  supersedes old eis_fits_script
;                                  and added a 'path_sep()' to the in_file
;-
;
pro eis_fits_script, in_dir, out_dir, logfile = logfile, $
                     noplan=noplan, nospcd=nospcd, noasrun=noasrun, $
                     rescued=rescued
;
; Defaults for making fits files. Change as defaults change!! 
; Current status is to DO doplan and dospcd, NOT dohk (never) or doasrun
; Turn off defaults with /noplan etc
;

 doplan=1
 dospcd=1
 dohk=0
 doasrun=0
 
 if n_elements(noplan) ne 0 then doplan=0
 if n_elements(nospcd) ne 0 then dospcd=0
 if n_elements(nohk) ne 0 then dohk=0
 if n_elements(noasrun) ne 0 then doasrun=0

  if n_elements(logfile) ne 0 then openw,lu,logfile,/append,/get_lun
  
  in_files = file_search(in_dir + path_sep() + 'eis_md*', count=count)
  if count eq 0 then begin
     !quiet = 0
     message, 'No files', /info
     !quiet = 1
     close,lu
     free_lun,lu
     return
  endif
  
  last_no_wrap_filename = 'eis_md_20190108_1228428105'
  first_wrap_filename   = 'eis_md_20190108_1234149331'

  for i = 0, count - 1 do begin  
     ; mcrw 20191119
     print
     if file_basename(in_files[i]) lt first_wrap_filename then oldseq=1 else oldseq=0

     ; mcrw 20190110
     !quiet=0
     message,'-----------', /info
     message,'Reformatting ' + in_files[i] + ' (' + strtrim(string(i),2) + ' out of ' + strtrim(string(count-1),2) + ')', /info
     !quiet=1

     ; mcrw 20210810
     if file_test(in_files[i]) then begin

        data=obj_new('eis_data', in_files[i], datasource='ccsds',hdr=hdr)
        exp_info = (hdr[0])->getexp_info()
        seq_id = exp_info.seq_id
        rast_id = exp_info.rast_id

        ; mcrw 20190110
        !quiet=0
        message,'seq_id = ' + strtrim(string(seq_id),2) + ', rast_id = ' + strtrim(string(rast_id),2),/info
        message, 'Calling eis_mkfits with oldseq = ' + strtrim(string(oldseq),2), /info
        !quiet=1

        eis_mkfits, data, hdr, $
                    doplan=doplan, dospcd=dospcd, doasrun=doasrun, $
                    fitsdir=out_dir,outfile=outfile, $
                    rescued=rescued, oldseq=oldseq, $
                    error_occurred

        parts = str_sep(in_files[i], '/')
        if n_elements(parts) ne 0 then infile = parts[n_elements(parts)-1] else infile=parts
      
        if error_occurred ne 0 then begin
           !quiet=0
           print,'Error occurred ('+strtrim(string(error_occurred),2)+'): '+infile+' '+outfile
           !quiet=1
           if n_elements(logfile) ne 0 then begin
;              parts = str_sep(in_files[i], '/')
;              if n_elements(parts) ne 0 then infile = parts[n_elements(parts)-1] else infile=parts
              printf,lu,'Error occurred ('+strtrim(string(error_occurred),2)+'): '+infile+' '+outfile
           endif

        endif

        if n_elements(logfile) ne 0 then begin
           parts = str_sep(in_files[i], '/')
           if n_elements(parts) ne 0 then infile = parts[n_elements(parts)-1] else infile=parts
           printf,lu,infile+' '+outfile
        endif
         
        obj_destroy,data
        obj_destroy,hdr

     endif else begin
        message, in_files[i] + ' not found'
     endelse

  endfor

  if n_elements(logfile) then begin
    close,lu
    free_lun,lu
  endif

  return
      
end
