#!/usr/bin/perl

#
# This program is scheduled to run on pg4 by cron but can also be run from the command line.
#

#
# allband = 0xF1
# antenna = 0xF0 merge
#           0xF1 allant
#
# Base start time on the first usc contact of the day in the contacts file for the day
# (20061029_contacts.txt appears not to have any usc passes - thats because it was a
# sunday...)
# 20061101_contacts.txt msp1 is first real, so maybe look for first real contact time
# and use that as the start time. Also sundays do not have any real time contacts...
#

use Time::Local;
use Date::Calc;
use Env qw(EGSE_SW_LIB);

# Instead of die-ing would be better to report the error in a file somewhere
# and then exit. This applies to all instances of die of course. Todo.

do "$EGSE_SW_LIB/date_calc.pl"     || die "Can't do date_calc.pl: $!\n";
do "$EGSE_SW_LIB/pipeline_defs.pl" || die "Can't do pipeline_defs.pl: $!\n";


# Open the master pipeline log if we can.
open(MLOG, ">> $pipeline_log/pipeline_log.txt") || die "Can't open $pipeline_log/$pipeline_log.txt for writing: $!";
print MLOG scalar(localtime), " (JST) daily_merge_mission1 started ";

# The start time will either be passed in (interactive mode)
# or will be generated from cron using todays date minus some days

$state = "cron"; # Default to cron job

$ant='usc34';	# Not needed for merge as sdtp will substitute merge antenna
$band=3;	# Merge telemetry
$mode='merge';	# Merge telemetry

# Set default values for start date, end date and start time to empty strings. If in
# interactive state these are not set then bail out.

$sdate = "";
$edate = "";
$stime = "";

# If there is anything on the command line then this is interactive mode
if($#ARGV != -1) {
    $state = "interactive";
    #
    # Get any command line arguments (must be in this order)
    #
    if(@ARGV) { $sdate  = shift; }
    if(@ARGV) { $edate  = shift; }
    if(@ARGV) { $stime 	= shift; }
    if(@ARGV) { $etime 	= shift; }
    #
    # Simple check for arguments. Could test actual times but sdtp will catch errors
    #
    die "No start date" if $sdate eq "";
    die "No end date"   if $edate eq "";
    die "No start time" if $stime eq "";
    $etime = $stime     if $etime eq "";
    #
    # All set, open the local log, update the master log and off we go
    #
    $pipeline_mission_log = "$pipeline_log/pipeline_mission_log_$sdate";
    open(LOG, ">> $pipeline_mission_log") || die "Can't open $pipeline_mission_log for writing: $!";
    print LOG "Started (manually) ", scalar(localtime), " (JST)\n";
    print MLOG "manually: daily_merge_mission1 $sdate $edate $stime $etime\n";

    $syear  = substr($sdate, 0, 4);
    $smonth = substr($sdate, 4, 2);
    $sday   = substr($sdate, 6, 2);
}
else {
    #
    # Started by cron. Calculate start date, end date and start time
    #
    print MLOG "by cron\n";

    #
    # Started by cron. Calculate start date, end date and start time
    #
    @today = today();	# @today[0..6] = seconds,minutes,hours,day,month,year,day_of_week (0 = sunday)

    #
    # If today is sunday then don't bother to do anything
    #
    if($today[6] == 0) {
	print MLOG "Sunday - not running\n";
	pipeline_exit("mission");
    }

    $start_day_offset = -7;
    $end_day_offset   = $start_day_offset + 1;
	
    #
    # If today is saturday then do saturday until monday morning
    #
    if($today[6] == 6) {	# Saturday == 6
        ++$end_day_offset;	# $end_day_offset should now be 5
    }

    #
    # Save defaults for start and end dates in case we can't get/parse orl files
    #
    @default_start_day = other_date(@today, $start_day_offset);		# Go back 7 days
    @default_end_day   = other_date(@today, $end_day_offset);		# Go back 5 or 6 days

    #
    # Extract strings from the dates
    #
    $default_syear  = sprintf "%04u", $default_start_day[5];	# Get starting year
    $default_smonth = sprintf "%02u", $default_start_day[4];	# Get starting month
    $default_sday   = sprintf "%02u", $default_start_day[3];	# Get starting day

    $default_eyear  = sprintf "%04u", $default_end_day[5];	# Get ending year
    $default_emonth = sprintf "%02u", $default_end_day[4];	# Get ending month
    $default_eday   = sprintf "%02u", $default_end_day[3];	# Get ending day

    #
    # Form start date string
    #
    $default_sdate = "$default_syear$default_smonth$default_sday";
    $default_edate = "$default_eyear$default_emonth$default_eday";

    #
    # Now try to parse start day orl file. If orl file not found for today, try tomorrow and lastly 3 days hence.
    # If all that fails then use today 10:00
    #
    $start_time_from_orl = 0;
    print MLOG "\tLooking for start time in orl file for $default_sdate \n";
##    foreach $attempt (0 .. 2) {
    @start_day = other_date(@today, $start_day_offset);		# Go back 7 days

    #
    # Extract strings from the dates
    #
    $syear  = sprintf "%04u", $start_day[5];	# Get starting year
    $smonth = sprintf "%02u", $start_day[4];	# Get starting month
    $sday   = sprintf "%02u", $start_day[3];	# Get starting day
    
    #
    # Form start date string
    #
    $sdate = "$syear$smonth$sday";
    
    #
    # Parse orl file
    #
    $stime_str = parse_orl_file($sdate, *MLOG);
    
    if(($stime_str ne "not found") && ($stime_str ne "parse error")) {
	$start_time_from_orl = 1;
##	    last;
    }
    elsif($stime_str eq "not found") {
	print MLOG "\torl file not found. Quitting.\n";
	pipeline_exit("mission");
    }
    elsif($stime_str eq "parse error") {
	print MLOG "\torl file parse error. Quitting.\n";
	pipeline_exit("mission");
    }
##    ++$start_day_offset;
#?        ++$end_day_offset;

##    }

    print "start_day_offset = $start_day_offset\n";

    #
    # Now try to parse end day orl file. If orl file not found for today, try tomorrow and lastly 3 days hence.
    # If all that fails then use today+1 10:00
    #
    $end_time_from_orl = 0;
    foreach $attempt (0 .. 2) {
	$end_day_offset = $start_day_offset + 1;
print "end_day_offset = $end_day_offset\n";
	@end_day   = other_date(@today, $end_day_offset);		# Go back 5 or 6 days

	#
	# Extract strings from the dates
	#
	$eyear  = sprintf "%04u", $end_day[5];	# Get ending year
	$emonth = sprintf "%02u", $end_day[4];	# Get ending month
	$eday   = sprintf "%02u", $end_day[3];	# Get ending day
	print MLOG "\tLooking for end time in orl file for $eyear$emonth$eday\n";

	#
	# Form end date strings
	#
	$edate = "$eyear$emonth$eday";
    
	#
	# Parse orl file
	#
	$etime_str = parse_orl_file($edate, *MLOG);

	if(($etime_str ne "not found") && ($etime_str ne "parse error")) {
	    $end_time_from_orl = 1;
	    last;
	}
	$start_day_offset += 1;

    }

    if(! $start_time_from_orl) {
	print MLOG "\tError parsing orl file - using default (1000) for start time\n";
	$stime_str = "1000";
    }
    else {
	print MLOG "\tStart time from orl file - $stime_str\n";
    }

    if(! $end_time_from_orl) {
	print MLOG "\tError parsing orl file - using default (1000) for end time\n";
	$etime_str = "1000";
    }
    else {
	print MLOG "\tStop time from orl file - $etime_str\n";
    }

    if(! $start_time_from_orl) {
	print MLOG "\tNo start orl file found, bailing\n";
	pipeline_exit("mission");
    }

    if((! $start_time_from_orl) || (! $end_time_from_orl)) {
	#
	# Extract strings from the dates
	#
	$syear  = sprintf "%04u", $default_start_day[5];	# Get starting year
	$smonth = sprintf "%02u", $default_start_day[4];	# Get starting month
	$sday   = sprintf "%02u", $default_start_day[3];	# Get starting day

	$eyear  = sprintf "%04u", $default_end_day[5];	# Get ending year
	$emonth = sprintf "%02u", $default_end_day[4];	# Get ending month
	$eday   = sprintf "%02u", $default_end_day[3];	# Get ending day

	#
	# Form start date string
	#
	$sdate = "$syear$smonth$sday";
	$edate = "$eyear$emonth$eday";

	$stime_str = "1000";
	$etime_str = "1000";
    }
    
    #
    # Open the local log if we can
    #
    $pipeline_mission_log = "$pipeline_log/pipeline_mission_log_$sdate";

    open(LOG, ">> $pipeline_mission_log") || die "Can't open $pipeline_mission_log for writing: $!";
    print LOG "Started ", scalar(localtime), " (JST)\n";
    print LOG "\tStart time $stime_str on $sdate\n";
    print LOG "\tEnd   time $etime_str on $edate\n";

    #
    # Addition when using parse_orl
    #
    $stime = $stime_str;
    $etime = $etime_str;

    #
    # Update master log and off we go
    #
    print MLOG "\tdaily_merge_mission1 $sdate $edate $stime $etime\n";
}

###die;

# Split the start date into constituent parts
$year  = substr $sdate, 0, 4;
$month = substr $sdate, 4, 2;
$day   = substr $sdate, 6, 2;

$date_string = "$year$month$day";

############################################### 0 - CLEAR OLD DATA

stage_title(*LOG, $state, "Removing old data in $merge_dir");

# Remove old ccsds packets
#perform(*LOG, $state, "/bin/rm -f $merge_dir/eis_*");
#perform(*LOG, $state, "/bin/rm -f $merge_dir/eis_*");
perform(*LOG, $state, "cd $merge_dir && /bin/rm -f eis_md_2*");
perform(*LOG, $state, "cd $merge_dir && /bin/rm -f eis_sts*");
perform(*LOG, $state, "cd $merge_dir && /bin/rm -f eis_dmp*");

# Remove old decompressed ccsds packets
#perform(*LOG, $state, "/bin/rm -f $merge_dir/decompressed/eis_*");
perform(*LOG, $state, "cd $merge_dir/decompressed && /bin/rm -f eis_md_2*");

# Remove old text files
perform(*LOG, $state, "/bin/rm -f $merge_dir/*.txt");

perform(*LOG, $state, "unset noclobber");


############################################### 1 - FETCH DATA

stage_title(*LOG, $state, "Fetching data");
#perform(*LOG, $state, "$HOME/bin/sdtp $mode $ant band=$band sdate=$sdate edate=$edate stime=$stime etime=$etime >&! $pipeline_log/md_fetch_$date_string.txt");
perform(*LOG, $state, "$HOME/bin/sdtp $mode $ant band=$band sdate=$sdate edate=$edate stime=$stime etime=$etime >& $pipeline_log" . "/md_fetch_" . "$date_string" . ".txt");

# Remove status files
perform(*LOG, $state, "/bin/rm $merge_dir/eis_sts*");


############################################### 2 - MISSION DATA CHECK

stage_title(*LOG, $state, "Mission data check");

@files = `ls $merge_dir/ | grep eis_md`;
if(! @files) {
    log_msg(*LOG, $state, "No ccsds mission data files");
    pipeline_exit("mission");
}
$log_msg_str = "Number of files: " . @files;
log_msg(*LOG, $state, $log_msg_str);

# Do the mission data header dump, which also updates the engineering.txt file
perform(*LOG, $state, "$HOME/bin/pipeline_md_hdr_dump $merge_dir > $merge_dir/md_hdr_check.txt");

# Do the ccsds check which detects missing packets, updating missing_packets.txt
perform(*LOG, $state, "$HOME/bin/pipeline_ccsds_check $merge_dir > $merge_dir/ccsds_hdr_check.txt");


###system(      "$HOME/bin/pipeline_ccsds_check $merge_dir $sdate $edate $stime $etime > $merge_dir/ccsds_hdr_check.txt");

# Move the files to DARTS
perform(*LOG, $state, "/bin/mv $merge_dir/md_hdr_check.txt $darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html");
perform(*LOG, $state, "/bin/mv $merge_dir/ccsds_hdr_check.txt $darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt");
perform(*LOG, $state, "/bin/cp $shutter_log/shutter.txt $staging/logs/");

#
# For now, if there are any archives with missing packets move them to the incomplete directory.
# Must handle these more sensibly in the future.
#
$number_missing = 0;
if(-e "$merge_dir/missing_packets.txt") {
    log_msg(*LOG, $state, "Missing packet file exists");
    open(BAD, "< $merge_dir/missing_packets.txt");
    while(<BAD>) {
	next if /^Missing/;
	last if /^$/;
	chomp;
	# TEMP!!!!
#	print STDERR "[$_]\n";

	log_msg(*LOG, $state, "Moving $merge_dir/$_ to $merge_dir/incomplete/");
	perform(*LOG, $state, "/bin/mv $merge_dir/$_ $merge_dir/incomplete/");
	++$number_missing;
    }
    close BAD;
    perform(*LOG, $state, "/bin/mv $merge_dir/missing_packets.txt $darts_mission/$year/$month/packet_check/missing_packets.$year$month$day.txt");
    log_msg(*LOG, $state, "Number of missing archives: $number_missing");
}
else {
    log_msg(*LOG, $state, "No missing packet file");
}

#
# Check for and record those files which are continuations of previous files.
# The decompression stage will move these files aside.
#
if(-e "$merge_dir/headless_packets.txt") {
    log_msg(*LOG, $state, "Headless packets file exists");
    open(BAD, "< $merge_dir/headless_packets.txt");
    while(<BAD>) {
	next if /^Headless/;
	last if /^$/;
	chomp;
	log_msg(*LOG, $state, "Moving $merge_dir/$_ to $merge_dir/incomplete/");
	perform(*LOG, $state, "/bin/mv $merge_dir/$_ $merge_dir/incomplete/") if(-e "$merge_dir/$_");
###	++$number_missing;
    }
    close BAD;
    perform(*LOG, $state, "/bin/mv $merge_dir/headless_packets.txt $darts_mission/$year/$month/packet_check/headless_packets.$year$month$day.txt");
}
else {
    log_msg(*LOG, $state, "No headless packets file");
}
#
# Check for and record those files which are not complete.
# The decompression stage will move these files aside.
#
if(-e "$merge_dir/incomplete_packets.txt") {
    log_msg(*LOG, $state, "Incomplete packets file exists");
    open(BAD, "< $merge_dir/incomplete_packets.txt");
    while(<BAD>) {
	next if /^Incomplete/;
	last if /^$/;
	chomp;
	log_msg(*LOG, $state, "Moving $merge_dir/$_ to $merge_dir/incomplete/");
	perform(*LOG, $state, "/bin/mv $merge_dir/$_ $merge_dir/incomplete/") if(-e "$merge_dir/$_");;
###	++$number_missing;
    }
    close BAD;
    perform(*LOG, $state, "/bin/mv $merge_dir/incomplete_packets.txt $darts_mission/$year/$month/packet_check/incomplete_packets.$year$month$day.txt");
}
else {
    log_msg(*LOG, $state, "No incomplete packets file");
}

perform(*LOG, $state, "/bin/mv $merge_dir/sequence_counts.txt $darts_mission/$year/$month/packet_check/sequence_counts_$year$month$day.txt");

# Get a count of number of mission data files here
#@pre_decompression = `ls $merge_dir/eis_md*`;
@pre_decompression = `ls $merge_dir/ | grep eis_md`;
$number_pre_decompression = @pre_decompression;
print STDOUT "Number of compressed files: ", scalar(@pre_decompression), "\n" if $state eq "interactive";
print LOG  "\tNumber of compressed files: ", scalar(@pre_decompression), "\n";


############################################### 3 - MISSION DATA DECOMPRESS

#stage_title(*LOG, $state, "Mission data check");

$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Starting mission data decompress\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Starting mission data decompress\n";

print LOG "\t(IDL) eis_md_decomp_fix1_script,'$merge_dir', /merge\n";

$output = <<EOF;
.comp $local_idl/eis_ccsds_interface__define

;.comp $local_idl/eis_md_decomp
;.comp $local_idl/eis_md_decomp_script
;eis_md_decomp_script,'$merge_dir', /merge

.comp $local_idl/mdppmdctrecover
;;;.comp $local_idl/mdppmdctrecover_merge

;.comp $local_idl/eis_md_decomp_fix
;.comp $local_idl/eis_md_decomp_fix_script
;eis_md_decomp_fix_script,'$merge_dir', /merge

.comp $local_idl/eis_md_decomp_fix1
.comp $local_idl/eis_md_decomp_fix1_script
eis_md_decomp_fix1_script,'$merge_dir', /merge
EOF

open(MD_DECOMP, "> $temp_idl/md_decomp.pro");
print MD_DECOMP $output;
close MD_DECOMP;

# Start the mission data decompress

###$date_string  = "$year$month$day";

#system("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_decomp.pro > /dev/null");
#system("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_decomp.pro > $pipeline_log/md_decompress_$date_string.txt");
system("$HOME/ssw_idl < $temp_idl/md_decomp.pro > $pipeline_log/md_decompress_$date_string.txt");

#system("remove_decomp_junk");

# Copy the decompression record and logs to DARTS
log_msg(*LOG, $state, "/bin/cp $HOME/work/localdata/log/decompression/merge_decomp_record.txt $darts_staging/logs/decompression/");
system("/bin/cp $HOME/work/localdata/log/decompression/merge_decomp_record.txt $darts_staging/logs/decompression/");

#log_msg(*LOG, $state, "/bin/mv $pipeline_log/md_decompress_$date_string.txt $darts_staging/logs/decompression/");
#system("/bin/mv $pipeline_log/md_decompress_$date_string.txt $darts_staging/logs/decompression/");

log_msg(*LOG, $state, "$HOME/bin/slim.pl <  $pipeline_log/md_decompress_$date_string.txt > $darts_staging/logs/decompression/md_decompress_$date_string.txt");
system("$HOME/bin/slim.pl < $pipeline_log/md_decompress_$date_string.txt > $darts_staging/logs/decompression/md_decompress_$date_string.txt");

# Get a count of number of mission data files here
#@post_decompression = `ls $merge_dir/decompressed/eis_md*`;
@post_decompression = `ls $merge_dir/decompressed/ | grep eis_md`;
$number_post_decompression = @post_decompression;
print STDOUT "Number of de-compressed files: ", scalar(@post_decompression), "\n" if $state eq "interactive";
print LOG  "\tNumber of de-compressed files: ", scalar(@post_decompression), "\n";


############################################### 4 - MISSION DATA REFORMAT

$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Starting mission data reformat\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Starting mission data reformat\n";
print LOG "\t(IDL) fits_script, '$merge_dir/decompressed/', '$temp_fits/'\n";

$translate_log_name = "md_translation" . "_$date_string" . ".txt";
$translate_log = "$pipeline_log/" . $translate_log_name;

$output = <<EOF;
.comp $local_idl/planning/jside_input/eis_read_orbit_events_reformatter
.comp $local_idl/eis_fits_calib
;;;.comp $local_idl/hw_help__define
.comp $local_idl/hw_data__define
.comp $local_idl/eis_data__define
.comp $local_idl/eis_data__readccsds
.comp $local_idl/eis_fits_dataid
.comp $local_idl/eis_spcd__define
;;;.comp $local_idl/eis_data__readfits
;;;.comp $local_idl/eis_modfits
;;;.comp $local_idl/eis_mkfits
;;;.comp $local_idl/eis_fits_script
eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/',logfile='$translate_log'
;eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/',logfile='$HOME/work/localdata/log/merge/md_translation.txt'
;eis_fits_script,'$merge_dir/decompressed/', '$temp_fits/',logfile='$HOME/work/localdata/log/merge/md_translation.txt', /nospcd
EOF

open(MD_FITS, "> $temp_idl/md_fits.pro");
print MD_FITS $output;
close MD_FITS;

$reformat_log = "$pipeline_log/md_reformat" . "_$date_string" . ".txt";
#$reformat_log = "/dev/null";

#print  "/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log\n";
#system("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log");
#print  "/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log\n";
#system("/san_darts/solar/ssw/gen/setup/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log");
perform(*LOG, $state, "$HOME/ssw_idl < $temp_idl/md_fits.pro >& $reformat_log");

perform(*LOG, $state, "/bin/cp $reformat_log $darts_staging/logs/md_reformat/");
perform(*LOG, $state, "/bin/cp $translate_log $darts_staging/logs/md_translation/");

# Get a count of number of mission data files here
#@post_reformat = `ls $temp_fits/eis_l0*`;
@post_reformat = `ls $temp_fits/ | grep eis_l0`;
print STDOUT "Number of reformatted files: ", scalar(@post_reformat), "\n" if $state eq "interactive";
print LOG  "\tNumber of reformatted files: ", scalar(@post_reformat), "\n";


############################################### 5 - UPDATE MD TRANSLATION DBASE

$current_time = scalar(localtime);
print STDOUT "$current_time (JST) - Update md translation database\n" if $state eq "interactive";
print LOG  "\n$current_time (JST) - Update md translation database\n";

# eis_mkfits writes all the ccsds packet filenames and the corresponding fits filenames processed
# to a text file. Open the text file and transfer the information to a perl database.

if(dbmopen(%trans, "$pipeline_log/mission_name_dbase", 0644)) {
    print STDOUT "mission_name_dbase opened\n" if $state eq "interactive";
    print LOG  "\tmission_name_dbase opened\n";
    if(open(REF_LOG, "< $translate_log")) {
	print STDOUT "Opened $translate_log ok\n" if $state eq "interactive";
	print LOG  "\tOpened $translate_log ok\n";
	while(<REF_LOG>) {
	    @files = split;		# Split ccsds filename and fits filename
	    $ccsds = $files[0];
	    $fits = $files[1];
	    $ccsds =~ s/ //g;		# Remove any white space from names
	    $fits =~ s/ //g;
	    chomp $fits;		# Remove newline if any
	    $trans{$ccsds} = $fits;
	    print LOG  "\tUpdating mission_name_database: $ccsds => $fits\n";
	    $temp{$ccsds} = $fits;
	}
	close(REF_LOG);
    }
    else {
	print STDOUT "Can't open $reformat_log: $!\n" if $state eq "interactive";
	print LOG  "\tCan't open $reformat_log: $!\n";
    }
    print STDOUT "Closing mission_name_dbase\n" if $state eq "interactive";
    print LOG  "\tClosing mission_name_dbase\n";
    if(dbmclose(%trans)) {
	print STDOUT "Closed mission_name_dbase ok\n" if $state eq "interactive";
	print LOG  "\tClosed mission_name_dbase ok\n";
    }
    else {
	print STDOUT "Closing mission_name_dbase failed: $!\n" if $state eq "interactive";
	print LOG  "\tClosing mission_name_dbase failed: $!\n";
    }
}
else {
    print STDOUT "Cannot open mission_name_dbase: $!\n" if $state eq "interactive";
    print LOG  "\tCannot open mission_name_dbase: $!\n";
}

# Copy the database to the staging area even if it was not updated
print STDOUT "/bin/cp $pipeline_log/mission_name_dbase $darts_staging/logs/\n" if $state eq "interactive";
print LOG  "\t/bin/cp $pipeline_log/mission_name_dbase $darts_staging/logs/\n";
system("/bin/cp $pipeline_log/mission_name_dbase $darts_staging/logs/");

goto skip1;

############################################### 6 - COMPRESS

# Now gzip everything in the temporary fits directory
# before moving to the DARTS directories

$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Gzip temporary fits files\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Gzip temporary fits files\n";
print LOG "\tgzip -f $temp_fits/*\n";

system("cd $temp_fits && gzip -f *");


############################################### 7 - COPY ENGINEERING FILES TO CALIBRATION AREA


$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Move engineering files to calibration area\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Move engineering files to calibration area\n";

# If the engineering.txt file exists then there is engineering data to be dealt with.
# Use the ccsds packet filename from the engineering.txt file to look up in the translation
# log to get the fits filename and use the engineering study type information to copy the
# fits file to the calibration directory.

$eng_dbase = 0;
if(-e "$merge_dir/engineering.txt") {
    print STDOUT "Engineering summary exists\n" if $state eq "interactive";
    print LOG "\tEngineering summary exists\n";
    $eng_dbase = 1 if(dbmopen(%eng_dbase, "$pipeline_log/engineering_name_dbase", 0644));
    if(dbmopen(%etrans, "$pipeline_log/mission_name_dbase", undef)) {
	if(open(ENG_LOG, "< $merge_dir/engineering.txt")) {
	    $line = <ENG_LOG>;				# Read title
	    while($line = <ENG_LOG>) {			# Read ccsds packet filename
		last if $line =~ /^$/;			# Finish on a blank line
		chop $line;				# Remove \n
		($path, $type) = split(/:/, $line);	# Get filename and type of engineering study
		print LOG "\t(path,type) = ($path, $type)\n";
		@path = split(/\//, $path);		# Split up filename path
		$ccsds_file = $path[-1];		# Get the ccsds filename
		$fits_file = $etrans{$ccsds_file};	# Get fits filename from ccsds filename
		$fits_file = $temp{$ccsds_file};	# Get fits filename from ccsds filename
		if($fits_file) {
		    $eng_dbase{$fits_file} = $type if $eng_dbase;	# Update engineering database
		    $destination = eng_path($type);		# Get where to copy the fits file and copy it
		    $source = $temp_fits . "/"  . $fits_file . ".gz";
		    if($destination ne "") {
			$dest1 = "$darts_mission" . $destination;
			$dest2 = "$soda_darts_mission" . $destination;
			print STDOUT "Moving $source to $dest1\n" if $state eq "interactive";
			print LOG  "\tMoving $source to $dest1 and $dest2\n";
			# mcrw 25/07/07 - Move instead of copy so engineering files don't appear in science area
			# mcrw 19/09/08 - Copy them now, not move
			system("/bin/cp $source $dest1");
			system("/bin/cp $source $dest2");
		    }
		    else {
			print STDOUT "No destination found for $fits_file ($type)\n" if $state eq "interactive";
			print LOG  "\tNo destination found for $fits_file ($type)\n";
		    }
		}
		else {
		    print STDOUT "No translation found for $ccsds_file\n" if $state eq "interactive";
		    print LOG  "\tNo translation found for $ccsds_file\n";
		}
	    }
	    close(ENG_LOG);
	}
	dbmclose(%etrans);
    }
    dbmclose(%eng_dbase) if($eng_dabse);
    system("/bin/mv $merge_dir/engineering.txt $darts_mission/$year/$month/packet_check/engineering.$year$month$day.txt");
    #
    # Generate cal study summary file from fits here?
    #  ./fits_cal_study.pl > ~/work/localdata/log/mission/cal_studies_list.txt
    #
}
else {
    print LOG "\tNo engineering summary file\n";
}
# Copy the engineering database to the staging area even if it was not updated
print STDOUT "/bin/cp $pipeline_log/engineering_name_dbase $darts_staging/logs/\n" if $state eq "interactive";
print LOG  "\t/bin/cp $pipeline_log/engineering_name_dbase $darts_staging/logs/\n";
system("/bin/cp $pipeline_log/engineering_name_dbase $darts_staging/logs/");


############################################### 8 - COPY MISSION FITS TO DARTS

$current_time = scalar(localtime);

print STDOUT $current_time, " (JST) - Moving mission data fits files to DARTS\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Moving mission data fits files to DARTS\n";

#
# Filenames:
#	eis_l0_yyyymmdd_hhmmss.fits
#

@files = `ls $temp_fits`;
foreach $file (@files) {
    next if !($file =~ /l0/);
    chomp $file;
    # Match destination directory with filename
    $darts_year  = substr $file,  7, 4;
    $darts_month = substr $file, 11, 2;
    $darts_day   = substr $file, 13, 2;

    print LOG "\t/bin/cp $temp_fits/$file $darts_mission/$darts_year/$darts_month/$darts_day/\n";
#    system("/bin/cp $temp_fits/$file $darts_mission/$darts_year/$darts_month/$darts_day/");
    system("/bin/cp $temp_fits/$file $soda_darts_mission/$darts_year/$darts_month/$darts_day/");
}

skip1:
############################################### 9 - CLEAN UP

$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Cleaning up\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Cleaning up\n";

# Remove temporary idl programs
print LOG "\t/bin/rm $temp_idl/*pro\n";
system("/bin/rm $temp_idl/*pro");

# Remove temporary fits files
print LOG "\t/bin/rm $temp_fits/*\n";
###system("/bin/rm $temp_fits/*");
system("cd $temp_fits && /bin/rm eis_l0*");

# Remove any temporary files from decompression
#print LOG "\t/bin/rm $eis_local/tmp/eis_md*\n";
###system("/bin/rm $eis_local/tmp/eis_md*");

perform(*LOG, $state, "mv $HOME/tmp/merge_record.txt $HOME/data/staging");
system("gzip -f $HOME/data/staging/merge_record.txt");

############################################### 10 - REMOVE QUICKLOOK MISSION DATA

$current_time = scalar(localtime);
print STDOUT $current_time, " (JST) - Removing quicklook mission data for $year$month$day\n" if $state eq "interactive";
print LOG "\n", $current_time, " (JST) - Removing quicklook mission data for $year$month$day\n";

# Remove directory in DARTS if it exists
if(-e "$darts_mission_ql/$year/$month/$day") {
    print LOG "\t/bin/rm -rf $darts_mission_ql/$year/$month/$day\n";
    system("/bin/rm -rf $darts_mission_ql/$year/$month/$day");
}
else {
    print LOG "\tDirectory $darts_mission_ql/$year/$month/$day does not exist\n";
}

# Remove md directory in sdtp if it exists
if(-e "$ql_mission/$year$month$day") {
    print LOG "\t/bin/rm -rf $ql_mission/$year$month$day\n";
    system("/bin/rm -rf $ql_mission/$year$month$day");
}
else {
    print LOG "\tDirectory $ql_mission/$year$month$day does not exist\n";
}

# Remove md fits directory in sdtp if it exists
if(-e "$ql_mission_fits/$year$month$day") {
    print LOG "\t/bin/rm -rf $ql_mission_fits/$year$month$day\n";
    system("/bin/rm -rf $ql_mission_fits/$year$month$day");
}
else {
    print LOG "\tDirectory $ql_mission_fits/$year$month$day does not exist\n";
}


############################################### 11 - Generate summary files

$ccsds_missing_file = "$darts_mission/$year/$month/packet_check/ccsds_hdrs.$year$month$day.txt";
$plan_file = "$staging/eis_plan/eis_plan_$syear$smonth$sday.txt";

$expected_number_of_rasters = `rasters_for_plan.sh $plan_file`;
chomp $expected_number_of_rasters;

perform(*LOG, $state, "estimate_rasters1.pl $plan_file > ~/tmp/rasters_for_plan.txt");

$plan_index = 0;
if(open PLN, "< $HOME/tmp/rasters_for_plan.txt") {
    while(<PLN>) {
	chomp;
	next if /^TL/;
	($sd, $st, $ed, $et) = split;
    }
    close PLN;
}

# Output format: filename	packets_per_exposure	number_of_exposures	timeline_id	reps_req	reps_done	RID	MID	Fx	Fy	ExpPerPos	data_volume	acronym
perform(*LOG, $state, "exposure_count.pl < $darts_mission/$year/$month/packet_check/md_hdrs.$year$month$day.html > $HOME/tmp/packets_per_exposure.txt");

# Output format: filename	packets_received	number_missing		number_of_gaps	date
perform(*LOG, $state, "ccsds_missing_summary1.pl $year$month$day $ccsds_missing_file > ~/tmp/missing.txt");
$total_ccsds_packets         = `awk '{ tot += \$2 } END { print tot }' ~/tmp/missing.txt`;
$total_ccsds_packets_missing = `awk '{ tot += \$3 } END { print tot }' ~/tmp/missing.txt`;
$total_gaps                  = `awk '{ tot += \$4 } END { print tot }' ~/tmp/missing.txt`;
$total_files                 = `grep eis_md ~/tmp/missing.txt | wc -l`;
chomp $total_files;
chomp $total_ccsds_packets;
chomp $totsl_ccsds_packets_missing;
chomp $total_gaps;

#
# put in exposures_missing_packets.pl output here, send to fold_reports.pl
#
# Output: filename	hdrs
perform(*LOG, $state, "exposures_missing_packets.pl < $ccsds_missing_file > $HOME/tmp/exp_missing.txt");

perform(*LOG, $state, "fold_reports.pl $HOME/tmp/missing.txt $HOME/tmp/packets_per_exposure.txt $translate_log $HOME/tmp/exp_missing.txt $HOME/tmp/reformat_summary.txt");
$total_exps_got = `awk '/^Filename/ { next } { tot += \$8 } END { print tot }' $HOME/tmp/reformat_summary.txt`;
chomp $total_exps_got;

#perform(*LOG, $state, "/bin/mv ~/tmp/missing.txt $staging/packet_summary/summary_$syear$smonth$sday.txt");

# Get total number of exposures expected and got
$total_exps_expected = `awk '{ tot += \$3 * \$11 } END { print tot }' $HOME/tmp/packets_per_exposure.txt`;
$total_exps_expected = -1 if $total_exps_expected == 0;

#if(open SUMMARY, "$staging/packet_summary/summary_$syear$smonth$sday.txt") {
if(open SUMMARY, "> $HOME/tmp/ref_sum.txt") {
    my $fits_num = scalar(@post_reformat);
    chomp $fits_num;
#    print SUMMARY "\n";
    print SUMMARY "Summary for the plan from $sd $st to $ed $et\n";
    print SUMMARY "\n";
    print SUMMARY "Rasters Planned          : ", sprintf("%6u", $expected_number_of_rasters), "\n";
    print SUMMARY "Rasters (files) Received : ", sprintf("%6u", $total_files), "\n";
    print SUMMARY "Number Of Damaged Files  : ", sprintf("%6u", $number_missing), "\n";
    print SUMMARY "File Damage Rate         : ", sprintf("%6.2f", ($number_missing / $total_files) * 100.0), "%\n";
    print SUMMARY "Files Pre Decompression  : ", sprintf("%6u", $number_pre_decompression), "\n";
    print SUMMARY "Files Post Decompression : ", sprintf("%6u", $number_post_decompression), "\n";
    print SUMMARY "Number Of Fits Files     : ", sprintf("%6u", $fits_num), "\n";
    print SUMMARY "Expected Exposures       : ", sprintf("%6u", $total_exps_expected), "\n";
    print SUMMARY "Actual Exposures         : ", sprintf("%6u", $total_exps_got), "\n";
    print SUMMARY "Exposure Loss            : ", sprintf("%6.2f", 100.0 - ($total_exps_got / $total_exps_expected) * 100.0), "%\n";; 
    print SUMMARY "Total Ccsds Packets      : ", sprintf("%6u", $total_ccsds_packets), "\n";
    print SUMMARY "Ccsds Packets Missing    : ", sprintf("%6u", $total_ccsds_packets_missing), "\n";
    print SUMMARY "Ccsds Packets Loss Rate  : ", sprintf("%6.2f", ($total_ccsds_packets_missing / $total_ccsds_packets) * 100.0), "%\n";
    print SUMMARY "Number Of Gaps           : ", sprintf("%6u", $total_gaps), "\n\n";
    close SUMMARY;
}

perform(*LOG, $state, "cat $HOME/tmp/ref_sum.txt $HOME/tmp/reformat_summary.txt > $staging/reformat_summary/summary_$syear$smonth$sday.txt");
#perform(*LOG, $state, "/bin/rm $HOME/tmp/ref_sum.txt $HOME/tmp/reformat_summary.txt $HOME/tmp/packets_per_exposure.txt $HOME/tmp/missing.txt $HOME/tmp/exp_missing.txt $HOME/tmp/rasters_for_plan.txt $HOME/tmp/counts.txt $HOME/tmp/fold.txt");
perform(*LOG, $state, "/bin/rm $HOME/tmp/ref_sum.txt $HOME/tmp/reformat_summary.txt $HOME/tmp/packets_per_exposure.txt $HOME/tmp/missing.txt $HOME/tmp/exp_missing.txt $HOME/tmp/rasters_for_plan.txt");


############################################### 12 - EXIT

###perform(*LOG, $state, "study_log.csh $syear $smonth");
perform(*LOG, $state, "set noclobber");
pipeline_exit("mission");


############################################### subs

sub parse_contact_file {
    my $date = shift;
    my $fh   = shift;
    my $line;
    my $src_file;
    my $file;
	
    # Parse contact file to get start time
    $src_file = "$date" . "_contacts.txt";
    $file = `/bin/ls $pass_list_dir/$src_file`;
    if($file eq "") {
	print $fh "\tCan't find contact file for $date\n";
	return "";
    }
    chomp $file;
    print $fh "\tUsing contact file $file for $date\n";
    $line = "";

    # Get the second real time contact if it exists
    # otherwise get the first
    if(open(PASS, "< $file")) {
    	while(<PASS>) {
	    if(/real/) {
		chomp;
		$line = $_;
###		last;
	    }
    	}
    }
    else {
    	print $fh "\tCan't open contacts file for $date ($!)\n";
    	return "";
    }
    if($line eq "") {
        print $fh "\tNo start time found in $file\n";
        return "";
    }
    return $line;
}

sub parse_orl_file {
    my $date = shift;	# Date of contact
    my $fh   = shift;	# Log file filehandle

    my $line;
    my $src_file;
    my $file;
	
    # Parse orl file to get start time. $orl_dir is defined in pipeline_defs.
    $src_file = "EIS_" . "$date" . "????.orl";
    $file = `/bin/ls $orl_dir/$src_file`;
    if($file eq "") {
	print $fh "\tCan't find orl file for $date\n";
	return "not found";
    }
    chomp $file;
    print $fh "\tUsing orl file $file for $date\n";
    $line = "";

    if(open(ORL, "< $file")) {
	while(<ORL>) {
	    # orl file format has changed, with extra STARTs before the obstbl dump start
	    # Skip past these.
	    next if /^START:OPOG_UPLOAD_TI/;
	    if(/^START/) {
		$line = $_;
		last;
	    }
	    next;
	}
	close(ORL);
    }
    else {
    	print $fh "\tCan't open orl file for $date ($!)\n";
    	return "not found";
    }

    if($line eq "") {
        print $fh "\tNo start time found in $file\n";
        return "parse error";
    }
    #
    # $line should be like: START:2007/02/13.10:00:00 { SEQ:EIS_LOAD_OBSTBL_REAL ;}
    # Return hhmm portion
    #
    $hh = substr $line, 17, 2;
    $mm = substr $line, 20, 2;

    return "parse error" unless $hh =~ m/([0-1][0-9]|2[0-3])/;
    return "parse error" unless $mm =~ m/[0-5][0-9]/;

    return "$hh$mm";
}
