/*
 * dctdecomp.c
 *
 * This subroutine is prepared for Solar-B MDP JPEG decompression tool
 * Independent JPEG Group's software is modified for this subroutine.
 *
 *  Revision History
 *     Ver. 0.1  first version   T.Shimizu   2001.06.29
 *
 *
 * Copyright (C) 1991-1997, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains a command-line user interface for the JPEG decompressor.
 * It should work on any system with Unix- or MS-DOS-style command lines.
 *
 * Two different command line styles are permitted, depending on the
 * compile-time switch TWO_FILE_COMMANDLINE:
 *	djpeg [options]  inputfile outputfile
 *	djpeg [options]  [inputfile]
 * In the second style, output is always to standard output, which you'd
 * normally redirect to a file or pipe to some other program.  Input is
 * either from a named file or from standard input (typically redirected).
 * The second style is convenient on Unix but is unhelpful on systems that
 * don't support pipes.  Also, you MUST use the first style if your system
 * doesn't do binary I/O to stdin/stdout.
 * To simplify script writing, the "-outfile" switch is provided.  The syntax
 *	djpeg [options]  -outfile outputfile  inputfile
 * works regardless of which command line style is used.
 */

#include "cdjpeg.h"		/* Common decls for cjpeg/djpeg applications */
#include "jversion.h"		/* for version message */

#include <ctype.h>		/* to declare isprint() */

/*
 *  For Solar-B application        T.Shimizu 2001.07.01
 *  External linkage between main and JFREAD_solarb
 */
unsigned char dct_inputBuffer[256*1024*2];
int dct_inputBuffersize;
int dct_outputBuffer[256*1024*2];
int dct_outputBuffersize;
int dct_outputBufferoffset=0;

/*
 * For Solar-B application
 * This is included for checking decompressed results.
 * Should be commented out
 *
 *extern int image_height;     
 *extern int image_width;    
 */

/* Create the add-on message string table. */
#define JMESSAGE(code,string)   string ,
static const char * const cdjpeg_message_table[] = {
#include "cderror.h"
  NULL
};

/*
 * This list defines the known output image formats
 * (not all of which need be supported by a given version).
 * You can change the default output format by defining DEFAULT_FMT;
 * indeed, you had better do so if you undefine PPM_SUPPORTED.
 */

typedef enum {
        FMT_BMP,                /* BMP format (Windows flavor) */
        FMT_GIF,                /* GIF format */
        FMT_OS2,                /* BMP format (OS/2 flavor) */
        FMT_PPM,                /* PPM/PGM (PBMPLUS formats) */
        FMT_RLE,                /* RLE format */
        FMT_TARGA,              /* Targa format */
        FMT_TIFF                /* TIFF format */
} IMAGE_FORMATS;

#ifndef DEFAULT_FMT             /* so can override from CFLAGS in Makefile */
#define DEFAULT_FMT     FMT_PPM
#endif

static IMAGE_FORMATS requested_fmt;

/*
 * Marker processor for COM and interesting APPn markers.
 * This replaces the library's built-in processor, which just skips the marker.
 * We want to print out the marker as text, to the extent possible.
 * Note this code relies on a non-suspending data source.
 */

LOCAL(unsigned int)
jpeg_getc (j_decompress_ptr cinfo)
/* Read next byte */
{
  struct jpeg_source_mgr * datasrc = cinfo->src;

  if (datasrc->bytes_in_buffer == 0) {
    if (! (*datasrc->fill_input_buffer) (cinfo))
      ERREXIT(cinfo, JERR_CANT_SUSPEND);
  }
  datasrc->bytes_in_buffer--;
  return GETJOCTET(*datasrc->next_input_byte++);
}


METHODDEF(boolean)
print_text_marker (j_decompress_ptr cinfo)
{
  boolean traceit = (cinfo->err->trace_level >= 1);
  INT32 length;
  unsigned int ch;
  unsigned int lastch = 0;

  length = jpeg_getc(cinfo) << 8;
  length += jpeg_getc(cinfo);
  length -= 2;			/* discount the length word itself */

  if (traceit) {
    if (cinfo->unread_marker == JPEG_COM)
      fprintf(stderr, "Comment, length %ld:\n", (long) length);
    else			/* assume it is an APPn otherwise */
      fprintf(stderr, "APP%d, length %ld:\n",
	      cinfo->unread_marker - JPEG_APP0, (long) length);
  }

  while (--length >= 0) {
    ch = jpeg_getc(cinfo);
    if (traceit) {
      /* Emit the character in a readable form.
       * Nonprintables are converted to \nnn form,
       * while \ is converted to \\.
       * Newlines in CR, CR/LF, or LF form will be printed as one newline.
       */
      if (ch == '\r') {
	fprintf(stderr, "\n");
      } else if (ch == '\n') {
	if (lastch != '\r')
	  fprintf(stderr, "\n");
      } else if (ch == '\\') {
	fprintf(stderr, "\\\\");
      } else if (isprint(ch)) {
	putc(ch, stderr);
      } else {
	fprintf(stderr, "\\%03o", ch);
      }
      lastch = ch;
    }
  }

  if (traceit)
    fprintf(stderr, "\n");

  return TRUE;
}


/* 
 * --------------------
 * dctdecomp subroutine
 * --------------------
 */

int
DctDecomp (inputBuffer, inputBuffersize, outputBuffer, outputBuffersize)
  unsigned char inputBuffer[];
  int inputBuffersize;
  int outputBuffer[];
  ////mcrw  int outputBuffersize;
  int *outputBuffersize;
{
  int image_width;
   int image_height;
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;

  register int i;

#ifdef PROGRESS_REPORT
  struct cdjpeg_progress_mgr progress;
#endif

  djpeg_dest_ptr dest_mgr = NULL;
  JDIMENSION num_scanlines;

  /* Initialize the JPEG decompression object with default error handling. */
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&cinfo);

  /* Add some application-specific error messages (from cderror.h) */
  jerr.addon_message_table = cdjpeg_message_table;
  jerr.first_addon_message = JMSG_FIRSTADDONCODE;
  jerr.last_addon_message = JMSG_LASTADDONCODE;

  /* Insert custom marker processor for COM and APP12.
   * APP12 is used by some digital camera makers for textual info,
   * so we provide the ability to display it as text.
   * If you like, additional APPn marker types can be selected for display,
   * but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   */
  jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
  jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  /* Now safe to enable signal catcher. */
#ifdef NEED_SIGNAL_CATCHER
  enable_signal_catcher((j_common_ptr) &cinfo);
#endif


#ifdef PROGRESS_REPORT
  start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  /*
   * Define input Buffer size for jpeg library 
   */
  dct_inputBuffersize= inputBuffersize;
  for (i=0;i<inputBuffersize;i++) dct_inputBuffer[i]= inputBuffer[i];


  /* Specify data source for decompression */
  jpeg_stdio_src(&cinfo);

  /* Read file header, set default decompression parameters */
  (void) jpeg_read_header(&cinfo, TRUE);

  /* Initialize the output module now to let it override any crucial
   * option settings (for instance, GIF wants to force color quantization).
   */
  /*
   * wrppm.c was modefied for Solar-B application
   */

  requested_fmt= FMT_PPM;
  dest_mgr = jinit_write_ppm(&cinfo);

  /* Start decompressor */
  (void) jpeg_start_decompress(&cinfo);


  /* Process data */
  while (cinfo.output_scanline < cinfo.output_height) {
    num_scanlines = jpeg_read_scanlines(&cinfo, dest_mgr->buffer,
					dest_mgr->buffer_height);
    (*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
  }
  /*
   * Above command restores decompressed data in dct_outputBuffer
   */ 

  /*
   * Added for Solar-B application   T.Shimizu   2001.6.29
   */
  dct_outputBuffersize= dct_outputBufferoffset;
  ///mcrw  outputBuffersize= dct_outputBuffersize;
  *outputBuffersize= dct_outputBuffersize;
  ///mcrw  for (i=0;i<outputBuffersize;i++) outputBuffer[i]= dct_outputBuffer[i];
  for (i=0;i<*outputBuffersize;i++) outputBuffer[i]= dct_outputBuffer[i];


  /*
   * for Solar-B application
   * This is for checking the decompressed data
   * Should be commeted out
   */
   image_width = cinfo.output_width;
   image_height = cinfo.output_height;

   // mcrw 20070129
   //   image_width = cinfo.output_height;
   //   image_height = cinfo.output_width;

   //   fprintf(stderr, "Width%dHeight%d\n", image_width, image_height);
#if 0
   fprintf(stderr, "Width %d, Height %d\n", image_width, image_height);
   fprintf(stderr, "output buffer size = %u\n", *outputBuffersize);
#endif
   /* */


#ifdef PROGRESS_REPORT
  /* Hack: count final pass as done in case finish_output does an extra pass.
   * The library won't have updated completed_passes.
   */
  progress.pub.completed_passes = progress.pub.total_passes;
#endif

  /* Finish decompression and release memory.
   * I must do it in this order because output module has allocated memory
   * of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   */
   (void) jpeg_finish_decompress(&cinfo);
   jpeg_destroy_decompress(&cinfo);
   

#ifdef PROGRESS_REPORT
  end_progress_monitor((j_common_ptr) &cinfo);
#endif

}
