#include        <inttypes.h>
/** 
 *  @file  
 *  @brief Time conversion functions used for telemetry 
 *         obtained from Data Distributors.

In the ISAS data flow, authorized time calibration is included in data from the Sirius database. There is another time database automatically created in daily operations. Latter database can be used with this Library.

 */

/** Low resolution TI clock
 * \var typedef uint32_t tilow_t; 
 * TI (free run clock of the Solar-B S/C) in the unit of 1/32 sec:
 * Note there is High resolution TI clock in the unit of 1/512 sec.
 */
typedef uint32_t tilow_t; 

/** Type to hold Solar-B time
 * \var typedef double sbtime_t; 
 * Here, Solar-B time is elapsed seconds from 2000/01/01 00:00:00 (UT)
 */
typedef double sbtime_t; 

/** Type to hold UT time. 
 * \var typedef uint8_t utbcd_t[10]; 
 * 10 bytes of BCD "YYYY0DDDhhmmssmmmuuu": Here DDD is total day.
 */
typedef uint8_t utbcd_t[10]; 

/** Initialize conversion library from TI to Solar-B time 
    (= load data base file).
 * @param[in] fname    File name of time conversion table
 * @param[in] sbtime0  Rough estimate of Solar-B time to be converted
 *                     ( used to resolve role over of TI clock)
 * @retval     0       Succeed.
 * @retval    -1       Failed (time range or file name).
 */
int SB_Time_Init( char *fname, sbtime_t sbtime0 );

/** Perform time conversion from TI to Solar-B time.
 * @param[in] tilow    Low resolution TI clock in the unit of 1/32 sec
 * @param[in] sbtime0  Rough estimate of Solar-B time to be converted
 *                     ( used to resolve role over of TI clock).
 * @return             Solar-B time converted from the TI.
 */
sbtime_t SB_Time_Convert( tilow_t tilow, sbtime_t sbtime0 );

/** Initialize conversion library between Solar-B time and UT
 * @param[in] fname    File name of leap second database.
 * @retval     0       Succeed.
 * @retval    -1       Failed (file name).
 */
int UtTt_Init( char *fname );

/** Convert Solar-B time to UT time
 * @param[in]  sbtime  Solar-B time
 * @param[out] utbcd   BCD encoded UT time converted from the Solar-B time
 * @retval     0       Succeed.
 * @retval    -1       Failed.
 */
int TttoUt( sbtime_t sbtime, utbcd_t utbcd );

/*! Convert UT time to Solar-B time
 * @param[in]  utbcd   BCD encoded UT time
 * @retval    >0       Solar-B time converted from the UT time
 * @retval    -1       Failed.
 */
double UttoTt( utbcd_t utbcd );
