/* there may be several versions of wrfits.c, recent ones should have
a date and a history */
/* 1/23/2004 - generalize wrfits to handle I*4 as well as I*2 arrays */
/* 1/12/2004 - adding keys as linked lists to make it easier to add
keys in programs that write FITS files, older versions used an array
of strings. That was fine (and faster) when the list didn't need to
be modified. The linked list has some more runtime overhead but is
easier to maintain and modify. */
#define SUCCESS_RETURN	0
#define ERROR_RETURN	1
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <time.h>
 /* the list definitions are in wrfits.h */
#include "wrfits.h"
 /* the recommended way to end lists is to have the end node point to
 itself. But a NULL works also, the self reference is convenient when extending
 to trees and other stuff not done here. */
 FITSkey listEnd = { NULL, NULL};
 int	ana_type_size[5] = {1,2,4,4,8};
 /*------------------------------------------------------------------------- */
FITSkey *newKeyList()
 {
 FITSkey *x;
 x = (FITSkey *) malloc(sizeof *x);
 x->name = NULL;
 x->next = &listEnd;		/* all use the same end */
 return x;			/* return the new head */
 }
  /*------------------------------------------------------------------------- */
void clearKeyList(FITSkey *node)
 {
 FITSkey *x = node, *y;
 if (node != NULL) {
   x = x->next;	  /* skip the head */
   /* read through and deallocate everybody (except head and tail) */
   if (x == NULL) { printf("problem in clearKeyList\n"); exit(1); }
   while (x->next) {
     //printf("name: %s\n", x->name);
     if (x->name != NULL) free(x->name);
     y = x;
     x = x->next;
     free(y);
   }
   node->next = &listEnd;		/* all use the same end */
   }
 return;
 }
 /*------------------------------------------------------------------------- */
void showList(FITSkey *node)
 {
 FITSkey *x = node, *y;
 if (node == NULL) { printf("NULL list\n"); return; }
 while (x->next) {
   printf("%s\n", x->name);
   x = x->next;
 }
 }
 /*------------------------------------------------------------------------- */
FITSkey *addKeyAfter(char *key, FITSkey *node)
 {
 FITSkey *x;
 x = (FITSkey *) malloc(sizeof *x);
 /* note that key has to have been a malloc'ed string or we will crash
 when this list is deleted. Also, key should not be free'ed after passing
 it here. */
 x->name = key;
 x->next = node->next;	/* steal the current next (often the end) */
 node->next = x;		/* make current before x, so x inserted after */
				/* the old current */
 return x;			/* return the new one */
 }
 /*------------------------------------------------------------------------- */
char * ana_time()
 /*returns current time in a string */
 {
 time_t	t;
 char	*p, *s;
 s = (char *) malloc(9);
 t = time(NULL);	p = ctime( &t );
 strncpy( s, p+11, 8);
 *( s + 8) = '\0';
 return s;
 }
 /*------------------------------------------------------------------------- */
char * ana_date()
 /*returns current date in a string */
 {
 time_t	t;
 char	*p, *s;
 s = (char *) malloc(13);
 t = time(NULL);	p = ctime( &t );
 strncpy( s, p+4, 6);	strncpy( s+6, ", ", 2);  strncpy( s+8, p+20, 4);
 *( s + 12) = '\0';
 return s;
 }
 /*------------------------------------------------------------------------- */
void swapb(char *x,int n)
 {
 int   i;
 char  xq;
 if (n < 2 ) { printf("error in swapb count, n = %d\n",n);  return; }
 if (n%2 != 0 ) n--;
 for (i=0;i<n;i += 2) { xq=x[i];  x[i] = x[i+1];  x[i+1] = xq; }
 }
 /*--------------------------------------------------------------------------*/
void swapl(x,n)
 char x[];
 int     n;	/* the number of longs (4 bytes each) to reverse */
 { int   i1,i2,i3,i4,i;
   char  xq;
 i1=0; i2=1; i3=2; i4=3;
 for (i=0;i<n;i++) { xq=x[i1];
	 x[i1]=x[i4];    x[i4]=xq;       xq=x[i2];
	 x[i2]=x[i3];    x[i3]=xq;
	 i1+=4;  i2+=4;  i3+=4;  i4+=4;  }
 }
 /*--------------------------------------------------------------------------*/
void swapd(x,n)
 char x[];
 int     n;	/* the number of F*8 (8 bytes each) to reverse */
 { int   i1,i2,i3,i4,i5,i6,i7,i8,i;
   char  xq;
 i1=0; i2=1; i3=2; i4=3; i5=4; i6=5; i7=6; i8=7;
 for (i=0;i<n;i++) {
 	xq=x[i1];	x[i1]=x[i8];    x[i8]=xq;
	xq=x[i2];	x[i2]=x[i7];    x[i7]=xq;
	xq=x[i3];	x[i3]=x[i6];	x[i6]=xq;
	xq=x[i4];	x[i4]=x[i5];	x[i5]=xq;
	i1+=8;  i2+=8;  i3+=8;  i4+=8; i5+=8; i6+=8; i7+=8; i8+=8;  }
 }
 /*------------------------------------------------------------------------- */
int wrfits(unsigned char *x, char *name, int ndim, int dimen[], FITSkey *keys, int dtype)
 /* generalize for I*4 as well as i*2, keys are now a linked list */
 /* assumes any additional keys are already formatted as an array
 of null terminated strings, each le 80 bytes long (we pad each
 card image up to the required 80) */
 {
 FILE *fopen(),*fout;
 int	bigendian, i, pads, nbsize, nout, nkeys;
 struct	tm *time_now;
 time_t tnow;
 FITSkey *xk;
 /* an endian detector, taken from SL's tiff library */
 { int one = 1; bigendian = (*(char *)&one == 0); }

 tnow = time(NULL);
 time_now = gmtime(&tnow);
 /* open the file */
 if ((fout=fopen(name,"w")) == NULL) {
     perror("FAILURE opening FITS file\n"); return ERROR_RETURN; }

 /* the first several are fixed since this must be 2-D I*2 */
 fprintf(fout,"SIMPLE  =                    T / created %24.24s%15s", ctime( &tnow)," ");
 switch (dtype) {
    case 0:  fprintf(fout,"BITPIX  =                    8%50s", " "); break;
    case 1:  fprintf(fout,"BITPIX  =                   16%50s", " "); break;
    case 2:  fprintf(fout,"BITPIX  =                   32%50s", " "); break;
    case 3:  fprintf(fout,"BITPIX  =                  -32%50s", " "); break;
    case 4:  fprintf(fout,"BITPIX  =                  -64%50s", " "); break;
    default:
     printf("illegal data type %d\n", dtype);
     fclose(fout);
     return ERROR_RETURN;
 }
 fprintf(fout,"NAXIS   =               %6d%50s", ndim, " ");
 nbsize = ana_type_size[dtype];
 {
   int j;
   for (j=0;j<ndim;j++) {
     fprintf(fout,"NAXIS%d  =               %6d%50s",j+1, dimen[j], " ");
     if (dimen[j] <= 0) {
     	printf("bad dim %d)\n", dimen[j]); return 1; }
     nbsize = nbsize * dimen[j];
   }
 }
 /* specialize for Solar B */
// fprintf(fout,"DATE    = '%02.2d/%02.2d/%04.4d'        /creation date%36s",time_now->tm_mday,
//   (time_now->tm_mon) + 1, time_now->tm_year + 1900 , " ");
 fprintf(fout,"DATE    = '%04.4d-%02.2d-%02.2dT%02.2d:%02.2d:%02.2d.000'        /creation date%23s",
   time_now->tm_year + 1900, (time_now->tm_mon) + 1, time_now->tm_mday, time_now->tm_hour,
   time_now->tm_min, time_now->tm_sec, " ");
 fprintf(fout,"DATE_RF0= '%04.4d-%02.2d-%02.2dT%02.2d:%02.2d:%02.2d.000'        /creation date%23s",
   time_now->tm_year + 1900, (time_now->tm_mon) + 1, time_now->tm_mday, time_now->tm_hour,
   time_now->tm_min, time_now->tm_sec, " ");
 /* we have 6 + ndim basal key lines plus whatever we got in keys, if more
 than 36 lines in all we have to make multiple header sets, note that
 we also have to count the END key so total is 7 + nkeys */
 nkeys = 0;
 /* 1/13/2004 - modified to use a linked list of keys */
 xk = keys;
 xk = xk->next;	  /* skip the head */
 while (xk->next) {
   nkeys++;
   //printf("%-80.80s\n", xk->name);
   fprintf(fout,"%-80.80s", xk->name);
   xk = xk->next;
 }
 fprintf(fout,"END%77s", " ");
 /* let's see where we are modulus 36, assuming 6 + ndim standard ones */
 pads = ((6 + ndim + nkeys - 1)/36) + 1;
 pads = 36 * pads;
 //printf("total lines to write = %d\n", pads);
 pads = pads - (6 + ndim) - nkeys;
 //printf("blank lines = %d\n", pads);
 if (pads > 0)  for (i=0;i<pads;i++) fprintf(fout,"%80s"," ");
 switch (dtype) {
   case 1:
     /* if we are running little endian, we should byte swap the image */
     if (!bigendian) swapb( (char *) x, nbsize);
     break;
   case 2:
     /* if we are running little endian, we should re-order the data */
     if (!bigendian) swapl( (char *) x, nbsize/4);
     break;
 }
 nout = fwrite(x, 1, nbsize, fout);
 if (nout != nbsize) {
     perror("FAILURE writing data to FITS file\n");
     fclose(fout);
     return ERROR_RETURN; }
 
 /* restore the swap if we are little endian */
 if (!bigendian) {
   switch (dtype) {
     case 1: swapb( (char *) x, nbsize);  break;
     case 2: swapl( (char *) x, nbsize/4);  break;
   }
 }
 fclose(fout);
 return	SUCCESS_RETURN;
 }
 /*------------------------------------------------------------------------- */
int wrfitsNS(unsigned char *x, char *name, int ndim, int dimen[], FITSkey *keys, int dtype,
  char *extend, int nextend)
 /* added arguments for optional extension, this is used for the image header
 for FG and SP solar B data files. We expect the EXTEND key for the primary header
 to be in keys already. Both extend and nextend must be non-zero or no extension. */
 /* this is a "no swap" (NS) version to let a little endian machine
 write bigendian fits files from a big endian data source w/o swapping
 twice, it will also work for the bigendian version of the reformatter */
 /* generalize for I*4 as well as i*2, keys are now a linked list */
 /* assumes any additional keys are already formatted as an array
 of null terminated strings, each le 80 bytes long (we pad each
 card image up to the required 80) */
 {
 FILE *fopen(),*fout;
 int	bigendian, i, pads, nbsize, nout, nkeys;
 struct	tm *time_now;
 time_t tnow;
 char *tnam;
 FITSkey *xk;
 /* an endian detector, taken from SL's tiff library */
 { int one = 1; bigendian = (*(char *)&one == 0); }

 tnow = time(NULL);
 time_now = gmtime(&tnow);
 tnam = rindex(name, '/');
 if (tnam) i = tnam + 1 - name; else i = 0;
 tnam = malloc(strlen(name) + 4);
 if (tnam==NULL) { perror("FAILed tnam malloc\n"); return ERROR_RETURN; }
 strcpy(tnam, name);
 tnam[i] = '.';
 /* open the file with a hidden name */
 if ((fout=fopen(tnam,"w")) == NULL) {
     perror("FAILURE opening FITS file\n"); return ERROR_RETURN; }

 /* the first several are fixed since this must be 2-D I*2 */
 fprintf(fout,"SIMPLE  =                    T / created %24.24s%15s", ctime( &tnow)," ");
 switch (dtype) {
    case 0:  fprintf(fout,"BITPIX  =                    8%50s", " "); break;
    case 1:  fprintf(fout,"BITPIX  =                   16%50s", " "); break;
    case 2:  fprintf(fout,"BITPIX  =                   32%50s", " "); break;
    case 3:  fprintf(fout,"BITPIX  =                  -32%50s", " "); break;
    case 4:  fprintf(fout,"BITPIX  =                  -64%50s", " "); break;
    default:
     printf("illegal data type %d\n", dtype);
     fclose(fout);
     return ERROR_RETURN;
 }
 fprintf(fout,"NAXIS   =               %6d%50s", ndim, " ");
 nbsize = ana_type_size[dtype];
 {
   int j;
   for (j=0;j<ndim;j++) {
     fprintf(fout,"NAXIS%d  =               %6d%50s",j+1, dimen[j], " ");
     if (dimen[j] <= 0) {
     	printf("bad dim %d)\n", dimen[j]); return 1; }
     nbsize = nbsize * dimen[j];
   }
 }
 /* specialize for Solar B level 0 data, this means a particular DATE format and a DATE_RF0 tag */
// fprintf(fout,"DATE    = '%02.2d/%02.2d/%04.4d'        /creation date%36s",time_now->tm_mday,
//   (time_now->tm_mon) + 1, time_now->tm_year + 1900 , " ");
 fprintf(fout,"DATE    = '%04.4d-%02.2d-%02.2dT%02.2d:%02.2d:%02.2d.000'        /creation date%23s",
   time_now->tm_year + 1900, (time_now->tm_mon) + 1, time_now->tm_mday, time_now->tm_hour,
   time_now->tm_min, time_now->tm_sec, " ");
 fprintf(fout,"DATE_RF0= '%04.4d-%02.2d-%02.2dT%02.2d:%02.2d:%02.2d.000'        /creation date%23s",
   time_now->tm_year + 1900, (time_now->tm_mon) + 1, time_now->tm_mday, time_now->tm_hour,
   time_now->tm_min, time_now->tm_sec, " ");
 /* we have 6 + ndim basal key lines plus whatever we got in keys, if more
 than 36 lines in all we have to make multiple header sets, note that
 we also have to count the END key so total is 7 + nkeys */
 nkeys = 0;
 /* 1/13/2004 - modified to use a linked list of keys */
 xk = keys;
 xk = xk->next;	  /* skip the head */
 while (xk->next) {
   nkeys++;
   //printf("%-80.80s\n", xk->name);
   fprintf(fout,"%-80.80s", xk->name);
   xk = xk->next;
 }
 fprintf(fout,"END%77s", " ");
 /* let's see where we are modulus 36, assuming 6 + ndim standard ones */
 pads = ((6 + ndim + nkeys - 1)/36) + 1;
 pads = 36 * pads;
 //printf("total lines to write = %d\n", pads);
 pads = pads - (6 + ndim) - nkeys;
 //printf("blank lines = %d\n", pads);
 if (pads > 0)  for (i=0;i<pads;i++) fprintf(fout,"%80s"," ");
 nout = fwrite(x, 1, nbsize, fout);
 if (nout != nbsize) {
     perror("FAILURE writing data to FITS file\n");
     fclose(fout);
     if (rename(tnam, name)) perror("FAILURE renaming FITS file\n");
     free(tnam);
     return ERROR_RETURN; }
 /* check if we have to add a "DUMP" extension */
 if (extend && nextend) {
   //printf("we have an extension, extend = %d, nextend = %d\n", extend, nextend);
   /* the extension has to start on a FITS "record" so we may have to pad with some
   zeroes here. Note that we were technically supposed to pad with zeroes at the
   end of the data array but that always seems like an unnecessary waste. */
   pads = 2880*( (nbsize-1)/2880 + 1) - nbsize;
   if (pads > 0) {
     /* there is probably a faster way to write out the zeroes */
     char *zips = calloc(pads, sizeof(char));
     nout = fwrite(zips, 1, pads, fout);
     if (nout != pads) {
       perror("FAILURE writing data to FITS file\n");
       fclose(fout);
       if (rename(tnam, name)) perror("FAILURE renaming FITS file\n");
       free(zips);
       free(tnam);
       return ERROR_RETURN; }
     free(zips);
   }
   /* we should now be on a FITS record boundary */
   fprintf(fout,"XTENSION= 'DUMP    '           / created %24.24s%15s", ctime( &tnow)," ");
   fprintf(fout,"BITPIX  =                    8%50s", " ");
   ndim = 1;
   fprintf(fout,"NAXIS   =               %6d%50s", ndim, " ");
   dimen[0] = nextend;
   {
     /* this could be simplified of course, just a copy of above even
     though we are just doing one dimension here */
     int j;
     for (j=0;j<ndim;j++) {
       fprintf(fout,"NAXIS%d  =               %6d%50s",j+1, dimen[j], " ");
       if (dimen[j] <= 0) {
     	  printf("bad dim %d)\n", dimen[j]); return 1; }
       nbsize = nbsize * dimen[j];
     }
   }
   fprintf(fout,"PCOUNT  =                    0%50s", " ");
   fprintf(fout,"GCOUNT  =                    1%50s", " ");
   fprintf(fout,"END%77s", " ");
   /* and pad to 36 lines (note that I have been using variable pads for several uses),
   these have to be blanks rather than zeros */
   pads = 29;
   for (i=0;i<pads;i++) fprintf(fout,"%80s"," ");
   /* and then the actual extension data */
   nout = fwrite(extend, 1, nextend, fout);
   //printf("nout for extend = %d\n", nout);
   if (nout != nextend) {
       perror("FAILURE writing data to FITS file\n");
       fclose(fout);
       if (rename(tnam, name)) perror("FAILURE renaming FITS file\n");
       free(tnam);
       return ERROR_RETURN; }
   /* add zero padding at the end */
   pads = 2880*( (nextend-1)/2880 + 1) - nextend;
   if (pads > 0) {
     /* there is probably a faster way to write out the zeroes */
     char *zips = calloc(pads, sizeof(char));
     nout = fwrite(zips, 1, pads, fout);
     if (nout != pads) {
       perror("FAILURE writing data to FITS file\n");
       fclose(fout);
       if (rename(tnam, name)) perror("FAILURE renaming FITS file\n");
       free(zips);
       free(tnam);
       return ERROR_RETURN; }
     free(zips);
   }
 }
 fclose(fout);
 if (rename(tnam, name)) perror("FAILURE renaming FITS file\n");
 free(tnam);
 return	SUCCESS_RETURN;
 }
 /*------------------------------------------------------------------------- */
FITSkey *code_key_i(char *key, int value, FITSkey *node)
 {
 /* returns a newly malloc'ed string of 30 char plus 0 (31), they can
 be free'd after they are written out */
 char	*s;
 s = (char *) malloc(32);
 snprintf(s, 32, "%-8.8s= %20d", key, value);
 return addKeyAfter(s, node);
 }
 /*------------------------------------------------------------------------- */
FITSkey *code_key_u(char *key, unsigned int value, FITSkey *node)
 {
 /* similar to  code_key_i but assumes unsigned */
 char	*s;
 s = (char *) malloc(32);
 snprintf(s, 32, "%-8.8s= %20u", key, value);
 return addKeyAfter(s, node);
 }
 /*------------------------------------------------------------------------- */
FITSkey *code_key_x(char *key, unsigned int value, FITSkey *node)
 {
 /* similar to  code_key_u but with hex format */
 char	*s;
 s = (char *) malloc(32);
 snprintf(s, 32, "%-8.8s= '0x%x'", key, value);
 return addKeyAfter(s, node);
 }
 /*------------------------------------------------------------------------- */
FITSkey *code_key_g(char *key, float value, FITSkey *node)
 {
 /* returns a newly malloc'ed string of 30 char plus 0 (31), they can
 be free'd after they are written out */
 char	*s;
 s = (char *) malloc(32);
 snprintf(s, 32, "%-8.8s= %20g", key, value);
 return addKeyAfter(s, node);
 }
 /*------------------------------------------------------------------------- */
FITSkey *code_key_s(char *key, char *value, FITSkey *node)
 {
 /* returns a newly malloc'ed string with a FITS string value, they can
 be free'd after they are written out */
 char	*s;
 int	len;
 s = (char *) malloc(84);
 /* we do a little checking and put the string in single quotes for the
 FITS standard */
 len = strlen(value);
 //printf("string: %s\n", value);
 if (len > 68) {
   printf("string too long in code_key_s,%d will be truncated\n",len);
   printf("string: %s\n", value);
   value[69] = 0; }
 snprintf(s, 84, "%-8.8s= '%s'", key, value);
 //printf("result: %s\n", s);
 return addKeyAfter(s, node);
 }
 /*------------------------------------------------------------------------- */
int modKey_s(char *key, char *value, FITSkey *node)
 /* modify a given key in the list, returns 0 if successful, 1 if not,
 this only search the list from the node above. For a full search, this
 must be the head of the list. If one knows a node may be changed, it can
 be saved and then used to reduce the time for the search. */
 {
 FITSkey *x = node;
 int	nc, len;
 char	lcopy[10], *p, *q, n, stat;
 nc = strlen(key);
 if (nc <= 0) return 1;		/* can't match a NULL in this scheme */
 nc = (nc<8)? nc:8;
 n = nc;  p = lcopy;  q = key;
 /* a FITS key is always 8 char long and blanks count. If the key passed here
 is less than 8, we have to pad it with blanks or we'll make mistakes. */
 while (n--) *p++ = *q++;
 if (nc < 8) { n = 8-nc;  while (n--) *p++ = ' '; }
 *p = 0;
 
 while (x->next) {
   stat = strncmp(lcopy, x->name, 8);
   if (stat == 0) {
     /* got it, now change the value */
     free(x->name);	/* because it may not be long enough */
     x->name = (char *) malloc(84);
     len = strlen(value);
     //printf("string: %s\n", value);
     if (len > 68) {
       printf("string too long in code_key_s,%d will be truncated\n",len);
       printf("string: %s\n", value);
       value[69	] = 0; }
     snprintf(x->name, 84, "%-8.8s= '%s'", key, value);
     return 0;
   }
   x = x->next;
 }
 return 1;			/* we never found it */
 }
 /*------------------------------------------------------------------------- */
int modKey_i(char *key, int value, FITSkey *node)
 /* modify a given key in the list, returns 0 if successful, 1 if not,
 this only searches the list from the node above. For a full search, this
 must be the head of the list. If one knows a node may be changed, it can
 be saved and then used to reduce the time for the search. */
 {
 FITSkey *x = node;
 int	nc;
 char	lcopy[10], *p, *q, n, stat;
 nc = strlen(key);
 if (nc <= 0) return 1;		/* can't match a NULL in this scheme */
 nc = (nc<8)? nc:8;
 n = nc;  p = lcopy;  q = key;
 /* a FITS key is always 8 char long and blanks count. If the key passed here
 is less than 8, we have to pad it with blanks or we'll make mistakes. */
 while (n--) *p++ = *q++;
 if (nc < 8) { n = 8-nc;  while (n--) *p++ = ' '; }
 *p = 0;
 
 while (x->next) {
   stat = strncmp(lcopy, x->name, 8);
   if (stat == 0) {
     /* got it, now change the value */
     free(x->name);	/* because it may not be long enough */
     x->name = (char *) malloc(32);
     snprintf(x->name, 32, "%-8.8s= %20d", key, value);
     return 0;
   }
   x = x->next;
 }
 return 1;			/* we never found it */
 }
 /*------------------------------------------------------------------------- */
int modKey_g(char *key, float value, FITSkey *node)
 /* modify a given key in the list, returns 0 if successful, 1 if not,
 this only search the list from the node above. For a full search, this
 must be the head of the list. If one knows a node may be changed, it can
 be saved and then used to reduce the time for the search. */
 {
 FITSkey *x = node;
 int	nc;
 char	lcopy[10], *p, *q, n, stat;
 nc = strlen(key);
 if (nc <= 0) return 1;		/* can't match a NULL in this scheme */
 nc = (nc<8)? nc:8;
 n = nc;  p = lcopy;  q = key;
 /* a FITS key is always 8 char long and blanks count. If the key passed here
 is less than 8, we have to pad it with blanks or we'll make mistakes. */
 while (n--) *p++ = *q++;
 if (nc < 8) { n = 8-nc;  while (n--) *p++ = ' '; }
 *p = 0;
 
 while (x->next) {
   stat = strncmp(lcopy, x->name, 8);
   if (stat == 0) {
     /* got it, now change the value */
     free(x->name);	/* because it may not be long enough */
     x->name = (char *) malloc(32);
     snprintf(x->name, 32, "%-8.8s= %20g", key, value);
     return 0;
   }
   x = x->next;
 }
 return 1;			/* we never found it */
 }
 /*------------------------------------------------------------------------- */
// int main()
//  {
//  /* test wrfits */
//  char	*fname = "testwrite.fits";
//  char	*keys[] = {"DUMMY01=      42","DUMMY02=           43"};
//  short	xx[600];
//  int	nx, ny, n, stat, nkeys, i;
//  time_t	t;
//  char	*p;
//  nkeys = sizeof(keys)/sizeof(char *);
// 
//  nx = 20;  ny = 30;
//  n = 20*30;
//  for (i=0;i<n;i++) xx[i] = i;
//  t = time(NULL);	p = ctime( &t );
//  printf("ctime: %s\n", p); 
//  printf("date: %s\n", ana_date());
//  printf("time: %s\n", ana_time());
//  printf("keys 0: %s\n", keys[0]);
//  printf("keys 1: %s\n", keys[1]);
//  printf("ready to call wrfits2d\n");
//  stat = wrfits2d(xx, fname, nx, ny, keys, nkeys);
//  printf("stat = %d\n", stat);
//  
//  }
