;+
; NAME: eis_status_sts1_conversions.pro
;
; PURPOSE: To convert EIS status 1 parameters from telemetered values
;          to engineering values
;      
; CATEGORY:
;
; CALLING SEQUENCE: 
;
; INPUTS: 
;
; KEYWORD PARAMETERS: None
;
; OUTPUTS: 
;
; CALLS:
;
; COMMON BLOCKS: None
;
; PROCEDURE:
;
; RESTRICTIONS: Uses SolarSoft
;
; MODIFICATION HISTORY:
;	03/June/2006 v1.0 mcrw wrote
;       15/Nov/2006  v1.1 mcrw minor corrections
;
;-------------------------------------------------------------------------------------------------
; files - string array of filenames in time order
; output - string pathname to where to write ps files
;
pro icu_currents, files, output, end_time

    p2v5_arr = eis_convert_icu_current_p2p5v  (eis_get_daily_data(files,'ICU_P2I5', 1, 2, maxp0, minp0, base_time, end_time))
    p5v_arr  = eis_convert_icu_current_p5v    (eis_get_daily_data(files,'ICU_P5I',  2, 2, maxp1, minp1))
    p15v_arr = eis_convert_icu_current_p15v   (eis_get_daily_data(files,'ICU_P15I', 3, 2, maxp2, minp2))
    n15v_arr = eis_convert_icu_current_n15v   (eis_get_daily_data(files,'ICU_N15I', 4, 2, maxp3, minp3))
    m28v_arr = eis_convert_icu_current_mbus28v(eis_get_daily_data(files,'MBUS_28I', 5, 2, maxp4, minp4))

    x = lindgen(n_elements(p2v5_arr)) * 2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/psu_currents.ps'
    !p.multi=[0,1,3]


    utplot,x,p2v5_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_current_p2p5v([minp0]),eis_convert_icu_current_p2p5v([maxp0])], $
      xstyle=2,pstyle=4, xticks=100, $
      title='PSU +2.5 V Current',ytitle='Amps',charsize=cs

    utplot,x,p5v_arr,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PSU +5 V Current',ytitle='Amps',charsize=cs,/ynozero

    utplot,x,p15v_arr,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PSU +15 V Current',ytitle='Amps',charsize=cs,/ynozero

    utplot,x,n15v_arr,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PSU -15 V Current',ytitle='Amps',charsize=cs,/ynozero

    utplot,x,m28v_arr,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PSU MBUS 28 V Current',ytitle='Amps',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0

end

pro icu_voltages, files, output, end_time

    p2v5_arr = eis_convert_icu_voltage_p2p5v  (eis_get_daily_data(files, 'ICU_P2V5', 1, 1, maxp1, minp1, base_time, end_time))
    p5v_arr  = eis_convert_icu_voltage_p5v    (eis_get_daily_data(files, 'ICU_P5V',  2, 1, maxp2, minp2))
    p15v_arr = eis_convert_icu_voltage_p15v   (eis_get_daily_data(files, 'ICU_P15V', 3, 1, maxp3, minp3))
    n15v_arr = eis_convert_icu_voltage_n15v   (eis_get_daily_data(files, 'ICU_N15V', 4, 1, maxp4, minp4))
    m28v_arr = eis_convert_icu_voltage_mbus28v(eis_get_daily_data(files, 'MBUS_28V', 5, 1, maxp5, minp5))
    
;    minr = min([minp1, minp2, minp3, minp4, minp5])
;    maxr = max([maxp1, maxp2, maxp3, maxp4, maxp5])

    x = lindgen(n_elements(p2v5_arr)) * 2

    cs=1.6

;    window, /free, xsize=1024
;    !p.multi=0
;    utplot,x,p2v5_arr,base_time,timerange=[base_time,end_time],yrange=[minr-1,maxr+1],xstyle=2,pstyle=4, $
;      title='PSU +2.5 Volts',ytitle='Volts',charsize=cs
;    utplot,x,p5v_arr,base_time,timerange=[base_time,end_time],yrange=[minr-1,maxr+1],xstyle=2,pstyle=4, $
;      title='PSU +5 Volts',ytitle='Volts',charsize=cs

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/psu_voltages.ps'
    !p.multi=[0,1,3]

    utplot,x,p2v5_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_voltage_p2p5v([minp1])-1.,eis_convert_icu_voltage_p2p5v([maxp1])+1.], $
      xstyle=3,pstyle=4,title='PSU +2.5 Volts',ytitle='Volts',charsize=cs

    utplot,x,p5v_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_voltage_p5v([minp2])-1.,eis_convert_icu_voltage_p5v([maxp2])+1.], $
      xstyle=3,pstyle=4,title='PSU +5 Volts',ytitle='Volts',charsize=cs

    utplot,x,p15v_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_voltage_p15v([minp3])-1.,eis_convert_icu_voltage_p15v([maxp3])+1.], $
      xstyle=3,pstyle=4,title='PSU +15 Volts',ytitle='Volts',charsize=cs

    utplot,x,n15v_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_voltage_n15v([minp4])-1.,eis_convert_icu_voltage_n15v([maxp4])+1.], $
      xstyle=3,pstyle=4,title='PSU -15 Volts',ytitle='Volts',charsize=cs

    utplot,x,m28v_arr,base_time,timerange=[base_time,end_time], $
      yrange=[eis_convert_icu_voltage_mbus28v([minp5])-1.,eis_convert_icu_voltage_mbus28v([maxp5])+1.], $
      xstyle=3,pstyle=4,title='PSU MBUS 28 Volts',ytitle='Volts',charsize=cs

    device,/close
    set_plot,'X'
    !p.multi=0

end

pro icu_temperatures, files, output, end_time

    p0 = eis_convert_icu_proc_temperature(eis_get_daily_data(files,'ICU_PROC_TEMP', 3, 3, maxp1, minp1, base_time, end_time))
    p1 = eis_convert_ccda_temperature(eis_get_daily_data    (files,'CCDA_TEMP',     1, 3, maxp2, minp2))
    p2 = eis_convert_ccdb_temperature(eis_get_daily_data    (files,'CCDB_TEMP',     2, 3, maxp3, minp3))

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_temperatures.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
;      yrange=[eis_convert_icu_current_p2p5v([minp1]),eis_convert_icu_current_p2p5v([maxp1])], $
      title='ICU PROC TEMP',ytitle='Degrees C',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD A TEMP',ytitle='Degrees C',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD B TEMP',ytitle='Degrees C',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_misc_parameters, files, output, end_time

    p0 = eis_get_daily_data(files,'ICU_SW_ID', 1, 4, maxp1, minp1, base_time, end_time)
    p1 = eis_get_daily_data(files,'PORT_READ', 2, 4, maxp2, minp2)
    p2 = eis_get_daily_data(files,'LAST_BC1_R', 3, 4, maxp3, minp3)
    p3 = eis_get_daily_data(files,'LAST_BC2_R', 3, 4, maxp3, minp3)
    p4 = eis_get_daily_data(files,'LAST_BC3_R', 3, 4, maxp3, minp3)
    p5 = eis_get_daily_data(files,'LAST_CMD_L_R', 3, 4, maxp3, minp3)
    p6 = eis_get_daily_data(files,'HC_TARGET_T', 3, 4, maxp3, minp3)
    p7 = eis_get_daily_data(files,'HC_DUTY_CYCLE', 3, 4, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_misc_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time], $
      xstyle=2,pstyle=4,title='ICU SW ID',ytitle='None',charsize=cs
      
    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PORT READ',ytitle='None',charsize=cs
    
    utplot,x,p2,base_time,timerange=[base_time,end_time],yrange=[-1,256], $
      xstyle=2,pstyle=4,title='LAST BC1 R',ytitle='None',charsize=cs
    
    utplot,x,p3,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='LAST BC2 R',ytitle='None',charsize=cs

    utplot,x,p4,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='LAST BC3 R',ytitle='None',charsize=cs

    utplot,x,p5,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='LAST CMD L R',ytitle='None',charsize=cs

    utplot,x,p6,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HC TARGET T',ytitle='None',charsize=cs

    utplot,x,p7,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HC DUTY CYCLE',ytitle='None',charsize=cs

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_status_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'EIS_MODE',             1,  5, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'CCD_BUF_TEST_STAT',    2,  5, maxp2, minp2)
    p2  = eis_get_daily_data(files,'CCD_BUF_TEST_ERR',     3,  5, maxp3, minp3)
    p3  = eis_get_daily_data(files,'CCD_BUF_TEST_WR_ERR',  4,  5, maxp3, minp3)
    p4  = eis_get_daily_data(files,'CCD_BUF_TEST_RD_ERR',  5,  5, maxp3, minp3)
    p5  = eis_get_daily_data(files,'ET_STAT',              6,  5, maxp3, minp3)
    p6  = eis_get_daily_data(files,'CMD_BUF_STAT',         7,  5, maxp3, minp3)
    p7  = eis_get_daily_data(files,'XRT_FF_STAT',          8,  5, maxp3, minp3)
    p8  = eis_get_daily_data(files,'EIS_FF_STAT',          9,  5, maxp3, minp3)
    p9  = eis_get_daily_data(files,'HM_MON_STAT',          10, 5, maxp3, minp3)
    p10 = eis_get_daily_data(files,'AEC_STAT',             11, 5, maxp3, minp3)
    p11 = eis_get_daily_data(files,'MEM_DMP_STAT',         12, 5, maxp3, minp3)
    p12 = eis_get_daily_data(files,'SEQ_STAT',             13, 5, maxp3, minp3)
    p13 = eis_get_daily_data(files,'MODE_EN_STAT',         14, 5, maxp3, minp3)
    p14 = eis_get_daily_data(files,'XRT_FF_REC',           15, 5, maxp3, minp3)
    p15 = eis_get_daily_data(files,'XRT_X_COR',            16, 5, maxp3, minp3)
    p16 = eis_get_daily_data(files,'XRT_Y_COR',            17, 5, maxp3, minp3)
    p17 = eis_get_daily_data(files,'MD_BUF_STAT',          18, 5, maxp3, minp3)
    p18 = eis_get_daily_data(files,'ICU_VF',               19, 5, maxp3, minp3)
    p19 = eis_get_daily_data(files,'PSU_VF',               20, 5, maxp3, minp3)
    p20 = eis_get_daily_data(files,'CAM_VF',               21, 5, maxp3, minp3)
    p21 = eis_get_daily_data(files,'MHC_VF',               22, 5, maxp3, minp3)
    p22 = eis_get_daily_data(files,'ASRC_STAT',            23, 5, maxp3, minp3)
    p23 = eis_get_daily_data(files,'MHC_LOAD_STAT',        24, 5, maxp3, minp3)
    p24 = eis_get_daily_data(files,'HC_STAT',              25, 5, maxp3, minp3)
    p25 = eis_get_daily_data(files,'EEPROM_STAT1',         26, 5, maxp3, minp3)
    p26 = eis_get_daily_data(files,'EEPROM_STAT2',         27, 5, maxp3, minp3)
    p27 = eis_get_daily_data(files,'EEPROM_COPY_REQ_STAT', 28, 5, maxp3, minp3)
    p28 = eis_get_daily_data(files,'EEPROM_COPY_PROG_STAT',29, 5, maxp3, minp3)
    p29 = eis_get_daily_data(files,'CMS_MHC_PWR_ON',       30, 5, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_status_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],yrange=[-1,6], $
      xstyle=2,pstyle=4,title='EIS MODE',ytitle='Mode',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD BUF TEST STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD BUF TEST ERR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p3,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD BUF TEST WR ERR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p4,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CCD BUF TEST RD ERR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p5,base_time,timerange=[base_time,end_time],yrange=[0,3], $
      xstyle=2,pstyle=4, $
      title='ET STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p6,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CMD BUF STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p7,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='XRT FF STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p8,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EIS FF STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p9,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM MON STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p10,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='AEC STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p11,base_time,timerange=[base_time,end_time],yrange=[0,4], $
      xstyle=2,pstyle=4,title='MEM DMP STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p12,base_time,timerange=[base_time,end_time],yrange=[0,4], $
      xstyle=2,pstyle=4,title='SEQ STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p13,base_time,timerange=[base_time,end_time],yrange=[0,4], $
      xstyle=2,pstyle=4,title='MODE EN STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p14,base_time,timerange=[base_time,end_time],yrange=[-1,2], $
      xstyle=2,pstyle=4,title='XRT FF REC',ytitle='None',charsize=cs,/ynozero

    utplot,x,p15,base_time,timerange=[base_time,end_time],yrange=[-1,256], $
      xstyle=2,pstyle=4,title='XRT X COR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p16,base_time,timerange=[base_time,end_time],yrange=[-1,256], $
      xstyle=2,pstyle=4,title='XRT Y COR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p17,base_time,timerange=[base_time,end_time],yrange=[-1,6], $
      xstyle=2,pstyle=4,title='MD BUF STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p18,base_time,timerange=[base_time,end_time],yrange=[0,3], $
      xstyle=2,pstyle=4,title='ICU VF',ytitle='None',charsize=cs,/ynozero

    utplot,x,p19,base_time,timerange=[base_time,end_time],yrange=[0,3], $
      xstyle=2,pstyle=4,title='PSU VF',ytitle='None',charsize=cs,/ynozero

    utplot,x,p20,base_time,timerange=[base_time,end_time],yrange=[0,3], $
      xstyle=2,pstyle=4,title='CAM VF',ytitle='None',charsize=cs,/ynozero

    utplot,x,p21,base_time,timerange=[base_time,end_time],yrange=[0,3], $
      xstyle=2,pstyle=4,title='MHC VF',ytitle='None',charsize=cs,/ynozero

    utplot,x,p22,base_time,timerange=[base_time,end_time],yrange=[0,3], $
xstyle=2,pstyle=4,title='ASRC STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p23,base_time,timerange=[base_time,end_time],yrange=[0,4], $
      xstyle=2,pstyle=4,title='MHC LOAD STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p24,base_time,timerange=[base_time,end_time],yrange=[0,4], $
      xstyle=2,pstyle=4,title='HC STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p25,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EEPROM STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p26,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EEPROM STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p27,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EEPROM COPY REQ STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p28,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EEPROM COPY PROG STAT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p29,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CAM_MHC_PWR_ON',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_psu_status_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'PSU_MARK', 1,  6, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'PSU_CCDB_HTR_EN', 2,  6, maxp2, minp2)
    p2  = eis_get_daily_data(files,'PSU_CCDA_HTR_EN', 3,  6, maxp3, minp3)
    p3  = eis_get_daily_data(files,'PSU_CCDB_BHTR_STAT', 4,  6, maxp3, minp3)
    p4  = eis_get_daily_data(files,'PSU_CCDA_BHTR_STAT', 5,  6, maxp3, minp3)
    p5  = eis_get_daily_data(files,'PSU_MHC_HTR_P28V', 6,  6, maxp3, minp3)
    p6  = eis_get_daily_data(files,'PSU_MHC_MECH_P28V', 7,  6, maxp3, minp3)
    p7  = eis_get_daily_data(files,'PSU_MHC_ELEC_P28V', 8,  6, maxp3, minp3)
    p8  = eis_get_daily_data(files,'PSU_MHC_MHTR', 9,  6, maxp3, minp3)
    p9  = eis_get_daily_data(files,'PSU_CAM_MHTR', 10, 6, maxp3, minp3)
    p10 = eis_get_daily_data(files,'PSU_CAM_P39V', 11, 6, maxp3, minp3)
    p11 = eis_get_daily_data(files,'PSU_CAM_N8V', 12, 6, maxp3, minp3)
    p12 = eis_get_daily_data(files,'PSU_CAM_P7V', 13, 6, maxp3, minp3)
    p13 = eis_get_daily_data(files,'PSU_CAM_P8V', 14, 6, maxp3, minp3)
    p14 = eis_get_daily_data(files,'PSU_CAM_P13V', 15, 6, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_psu_status_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],yrange=[-1,2], $
      xstyle=2,pstyle=4, $
      title='PSU MARK',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CCDB HTR EN',ytitle='None',charsize=cs

    utplot,x,p2,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CCDA HTR EN',ytitle='None',charsize=cs

    utplot,x,p3,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CCDB BHTR STAT',ytitle='None',charsize=cs

    utplot,x,p4,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CCDA BHTR STAT',ytitle='None',charsize=cs

    utplot,x,p5,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU MHC HTR P28V',ytitle='None',charsize=cs

    utplot,x,p6,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU MHC MECH P28V',ytitle='None',charsize=cs

    utplot,x,p7,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU MHC ELEC P28V',ytitle='None',charsize=cs

    utplot,x,p8,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU MHC MHTR',ytitle='None',charsize=cs

    utplot,x,p9,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM MHTR',ytitle='None',charsize=cs

    utplot,x,p10,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM P39V',ytitle='None',charsize=cs

    utplot,x,p11,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM N8V',ytitle='None',charsize=cs

    utplot,x,p12,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM P7V',ytitle='None',charsize=cs

    utplot,x,p13,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM P8V',ytitle='None',charsize=cs

    utplot,x,p14,base_time,timerange=[base_time,end_time],yrange=[-1,2],xstyle=2,pstyle=4, $
      title='PSU CAM P13V',ytitle='None',charsize=cs

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_hm_status_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'HM_OOL_ALERT', 1,  7, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'HM_PSU_TO', 2,  7, maxp2, minp2)
    p2  = eis_get_daily_data(files,'HM_CAM_TO', 3,  7, maxp3, minp3)
    p3  = eis_get_daily_data(files,'HM_MHC_TO', 4,  7, maxp3, minp3)
    p4  = eis_get_daily_data(files,'HM_PSU_OOL', 5,  7, maxp3, minp3)
    p5  = eis_get_daily_data(files,'HM_CAM_OOL', 6,  7, maxp3, minp3)
    p6  = eis_get_daily_data(files,'HM_MHC_OOL', 7,  7, maxp3, minp3)
    p7  = eis_get_daily_data(files,'HM_PARM_ID', 8,  7, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_hm_status_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
;      yrange=[eis_convert_icu_current_p2p5v([minp1]),eis_convert_icu_current_p2p5v([maxp1])], $
      title='HM OOL ALERT',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM PSU TO',ytitle='None',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM CAM TO',ytitle='None',charsize=cs,/ynozero

    utplot,x,p3,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM MHC TO',ytitle='None',charsize=cs,/ynozero

    utplot,x,p4,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM PSU OOL',ytitle='None',charsize=cs,/ynozero

    utplot,x,p5,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM CAM OOL',ytitle='None',charsize=cs,/ynozero

    utplot,x,p6,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM MHC OOL',ytitle='None',charsize=cs,/ynozero

    utplot,x,p7,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HM PARM ID',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_interface_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'CMD_IF_STAT1', 1,  8, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'MD_IF_STAT1', 2,  8, maxp2, minp2)
    p2  = eis_get_daily_data(files,'STAT_IF_STAT1', 3,  8, maxp3, minp3)
    p3  = eis_get_daily_data(files,'WD_IF_STAT1', 4,  8, maxp3, minp3)
    p4  = eis_get_daily_data(files,'CMD_IF_STAT2', 5,  8, maxp3, minp3)
    p5  = eis_get_daily_data(files,'MD_IF_STAT2', 6,  8, maxp3, minp3)
    p6  = eis_get_daily_data(files,'STAT_IF_STAT2', 7,  8, maxp3, minp3)
    p7  = eis_get_daily_data(files,'WD_IF_STAT2', 8,  8, maxp3, minp3)
    p8  = eis_get_daily_data(files,'MHC_422_STAT1', 9,  8, maxp3, minp3)
    p9  = eis_get_daily_data(files,'ROE_IF_STAT1', 10, 8, maxp3, minp3)
    p10 = eis_get_daily_data(files,'ROE_IF_STAT2', 11, 8, maxp3, minp3)
    p11 = eis_get_daily_data(files,'HSL_IF_STAT1', 12, 8, maxp3, minp3)
    p12 = eis_get_daily_data(files,'HSL_IF_STAT2', 13, 8, maxp3, minp3)
    p13 = eis_get_daily_data(files,'MHC_422_STAT2', 14, 8, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_interface_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
;      yrange=[eis_convert_icu_current_p2p5v([minp1]),eis_convert_icu_current_p2p5v([maxp1])], $
      title='CMD IF STAT1',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MD IF STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='STAT IF STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p3,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='WD IF STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p4,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CMD IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p5,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MD IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p6,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='STAT IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p7,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='WD IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p8,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MHC 422 STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p9,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='ROE IF STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p10,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='ROE IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p11,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HSL IF STAT1',ytitle='None',charsize=cs,/ynozero

    utplot,x,p12,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HSL IF STAT2',ytitle='None',charsize=cs,/ynozero

    utplot,x,p13,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MHC 422 STAT2',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_error_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'TC_FAILED_EC', 1,  9, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'CMD_IF_ERROR', 2,  9, maxp2, minp2)
    p2  = eis_get_daily_data(files,'PSU_STAT_ERROR', 3,  9, maxp3, minp3)
    p3  = eis_get_daily_data(files,'TC_FAILED_CMD_ID', 4,  9, maxp3, minp3)
    p4  = eis_get_daily_data(files,'ICU_ERROR_F', 5,  9, maxp3, minp3)
    p5  = eis_get_daily_data(files,'XRT_ERROR', 6,  9, maxp3, minp3)
    p6  = eis_get_daily_data(files,'HC_DUTY_ERROR', 7,  9, maxp3, minp3)
    p7  = eis_get_daily_data(files,'HC_PSU_TO', 8,  9, maxp3, minp3)
    p8  = eis_get_daily_data(files,'MDP_LL_ERROR', 9,  9, maxp3, minp3)
    p9  = eis_get_daily_data(files,'MHC_CMD_H', 10, 9, maxp3, minp3)
    p10 = eis_get_daily_data(files,'FT_ERROR', 11, 9, maxp3, minp3)
    p11 = eis_get_daily_data(files,'AEC_WIN_ERROR', 12, 9, maxp3, minp3)
    p12 = eis_get_daily_data(files,'AEC_PARMS_ERROR', 13, 9, maxp3, minp3)
    p13 = eis_get_daily_data(files,'AEC_TIME_ERROR', 14, 9, maxp3, minp3)
    p14 = eis_get_daily_data(files,'CAM_IF_ERROR', 15, 9, maxp3, minp3)
    p15 = eis_get_daily_data(files,'SEQ_ABORT_CODE', 16, 9, maxp3, minp3)
    p16 = eis_get_daily_data(files,'CMD_ID_FAILED_INT', 17, 9, maxp3, minp3)
    p17 = eis_get_daily_data(files,'MHC_IF_ERROR', 18, 9, maxp3, minp3)
    p18 = eis_get_daily_data(files,'EEPROM_ERROR', 19, 9, maxp3, minp3)
    p19 = eis_get_daily_data(files,'ET_ERROR', 20, 9, maxp3, minp3)
    p20 = eis_get_daily_data(files,'TC_FAILED_PKTC', 21, 9, maxp3, minp3)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_error_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
;      yrange=[eis_convert_icu_current_p2p5v([minp1]),eis_convert_icu_current_p2p5v([maxp1])], $
      title='TC_FAILED_EC',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CMD_IF_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='PSU_STAT_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p3,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='TC_FAILED_CMD_ID',ytitle='None',charsize=cs,/ynozero

    utplot,x,p4,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='ICU_ERROR_F',ytitle='None',charsize=cs,/ynozero

    utplot,x,p5,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='XRT_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p6,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HC_DUTY_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p7,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='HC_PSU_TO',ytitle='None',charsize=cs,/ynozero

    utplot,x,p8,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MDP_LL_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p9,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MHC_CMD_H',ytitle='None',charsize=cs,/ynozero

    utplot,x,p10,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='FT_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p11,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='AEC_WIN_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p12,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='AEC_PARMS_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p13,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='AEC_TIME_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p14,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CAM_IF_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p15,base_time,timerange=[base_time,end_time],yrange=[-1,16],xstyle=2,pstyle=4, $
      title='SEQ_ABORT_CODE',ytitle='None',charsize=cs,/ynozero

    utplot,x,p16,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='CMD_ID_FAILED_INT',ytitle='None',charsize=cs,/ynozero

    utplot,x,p17,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='MHC_IF_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p18,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='EEPROM_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p19,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='ET_ERROR',ytitle='None',charsize=cs,/ynozero

    utplot,x,p20,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='TC_FAILED_PKTC',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_counter_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'MDP_TIME',       1,  10, maxp0, minp0, base_time, end_time)
    p1  = eis_get_daily_data(files,'STATUS_PC',      2,  10, maxp1, minp1)
    p2  = eis_get_daily_data(files,'TC_REC_PKTC',    3,  10, maxp2, minp2)
    p3  = eis_get_daily_data(files,'RASTER_RUN_REM', 4,  10, maxp3, minp3)
    p4  = eis_get_daily_data(files,'SEQ_RUN_REM',    5,  10, maxp4, minp4)
    p5  = eis_get_daily_data(files,'EXP_NUM',        6,  10, maxp5, minp5)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_counter_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],yrange=[minp0-1,maxp0+1],xstyle=2,pstyle=4, $
      title='MDP_TIME',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],yrange=[minp1-1,maxp1+1],xstyle=2,pstyle=4, $
      title='STATUS_PC',ytitle='None',charsize=cs

    utplot,x,p2,base_time,timerange=[base_time,end_time],yrange=[minp2-1,maxp2+1],xstyle=2,pstyle=4, $
      title='TC_REC_PKTC',ytitle='None',charsize=cs

    utplot,x,p3,base_time,timerange=[base_time,end_time],yrange=[minp3-1,maxp3+1],xstyle=2,pstyle=4, $
      title='RASTER_RUN_REM',ytitle='None',charsize=cs

    utplot,x,p4,base_time,timerange=[base_time,end_time],yrange=[minp4-1,maxp4+1],xstyle=2,pstyle=4, $
      title='SEQ_RUN_REM',ytitle='None',charsize=cs

    utplot,x,p5,base_time,timerange=[base_time,end_time],yrange=[minp5-1,maxp5+1],xstyle=2,pstyle=4, $
      title='EXP_NUM',ytitle='None',charsize=cs

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_science_status_parameters, files, output, end_time

    p0  = eis_get_daily_data(files,'SEQ_I', 1,  11, maxp0, minp0, base_time, end_time)
    p1  = eis_get_daily_data(files,'SEQ_P', 2,  11, maxp1, minp1)
    p2  = eis_get_daily_data(files,'LL_I',  3,  11, maxp2, minp2)

    x = lindgen(n_elements(p0))*2

    cs=1.6


    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_science_status_parameters.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],yrange=[minp0-1,maxp0+1],xstyle=2,pstyle=4, $
      title='SEQ_I',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],yrange=[minp1-1,maxp1+1],xstyle=2,pstyle=4, $
      title='SEQ_P',ytitle='None',charsize=cs,/ynozero

    utplot,x,p2,base_time,timerange=[base_time,end_time],yrange=[minp2-1,maxp2+1],xstyle=2,pstyle=4, $
      title='LL_I',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro icu_spacecraft, files, output, end_time

    p0  = eis_get_daily_data(files,'TIME', 1,  12, maxp1, minp1, base_time, end_time)
    p1  = eis_get_daily_data(files,'SEQ_COUNTER', 2,  12, maxp2, minp2)

    x = lindgen(n_elements(p0))*2

    cs=1.6

    set_plot,'ps'
    device,xoff=2.5,yoff=6,xs=16,ys=18,file=output+'/icu_spacecraft.ps'
    !p.multi=[0,1,3]

    utplot,x,p0,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
;      yrange=[eis_convert_icu_current_p2p5v([minp1]),eis_convert_icu_current_p2p5v([maxp1])], $
      title='TIME',ytitle='None',charsize=cs

    utplot,x,p1,base_time,timerange=[base_time,end_time],xstyle=2,pstyle=4, $
      title='SEQ_COUNTER',ytitle='None',charsize=cs,/ynozero

    device,/close
    set_plot,'X'
    !p.multi=0
        
end

pro eis_status_sts1_conversions, dir, output, end_time

    files = findfile(dir + 'eis_sts1_*fits', count=count)
    if count eq 0 then begin
        print, 'eis_status_sts1_conversions : No files'
        return
    endif
    
    icu_voltages,                  files, output, end_time
    icu_currents,                  files, output, end_time
    icu_temperatures,              files, output, end_time
    icu_misc_parameters,           files, output, end_time
    icu_status_parameters,         files, output, end_time
    icu_psu_status_parameters,     files, output, end_time
    icu_hm_status_parameters,      files, output, end_time
    icu_interface_parameters,      files, output, end_time
    icu_error_parameters,          files, output, end_time
    icu_counter_parameters,        files, output, end_time
    icu_science_status_parameters, files, output, end_time
    icu_spacecraft,                files, output, end_time

end
