;+
; NAME:
;       EIS_FITS_COORD
;
; PURPOSE:
;	Adds keywords related to coordinates (pointing, SC attitude etc.) 
;	to the primary header of the EIS FITS file.
;
;
; CATEGORY:
;       Data Processing Software
;
; CALLING SEQUENCE:
;       eis_fits_coord, header, md_hdr
;
; INPUTS:
;       header: The primary header which will have keywords added to it.
;	md_hdr: EIS Mission Data header object array
;       plan: EIS planning object -> NB taken out! as it is not needed.
;       cal: EIS calibration object
;
; KEYWORD PARAMETERS:
;       none
;
;
; OUTPUTS:
;       The extended primary header
;
; CALLS:
;	fxhread
;	fxaddpar
;
; COMMON BLOCKS:
;
;
; PROCEDURE:
;
; RESTRICTIONS:
;
;
; MODIFICATION HISTORY:
;       November 2002: ivind Wikstl.
;       august 2006: Changed pointing from plan to spcd 
;                     CRPIX still uncertain where they will come from. Hardcoded to 0!!!
;       14 november 2006: Viggo Hansteen - defined crpix to (1,512)
;       16 november 2006: Viggo Hansteen - pointing information from spcd
;                                          moved 'tr_mode' to spcd from hk
;       18-Feb-2010  V. Hansteen     - Implemented John Mariska's 
;                                      "EIS pointing" ver 0.95 document.
;
; $Id: eis_fits_coord.pro,v 1.9 2010/02/18 10:44:24 viggoh Exp $
;-
pro eis_fits_coord, header, md_hdr, data, spcd, cal
;
  exp_info = md_hdr[0]-> getexp_info()
; 
  crpix1 = 1
  crpix2 = 1
; pixel sizes - should be standard or get from calibration data?
  cdelt1 = -1.*data->getdx_size() ; (arcsec pr pixel in solar-x)
  cdelt2 = 1.0 ; (arcsec pr pixel in solar-y)
  cdelt3 = 0.0223 ; (angstrom pr pixel)
; first compute FOVX and FOVY (as planned)
  yw = exp_info.yw
  nraster = 1
  nhdr = n_elements(md_hdr)
  nexp = exp_info.nexp
  nexp_prp = exp_info.exp_prp
  if exp_info.fmir_step gt 0 then $
    nraster = nexp/nexp_prp ; nr of unique raster positions
  fovx = -1.*nraster*cdelt1
  fovy = yw*cdelt2
; pointing of reference pixel (should get from Spacecraft data - TBC)
; 20100218 viggoh: now set to SW corner of image as measured at start
;                  of observation (or first good pointing after start 
;                  of obsrvation)
  crval1 = (spcd-> getcrval1())-cdelt1*((nraster+1)/2.-crpix1)+ $
     ( exp_info.cmirr_pos-cal->getxmidcmirrpos())*(cal->getcmirr_ss())+ $
     (cal->getslit_pointing_offset(data->getslit_ind()))
  crval2 = (spcd-> getcrval2())-(cal->getymidslitpos())+(data->getys())[0]
; defintions of data units for axis 1 2 3
  cunit1 = 'arcsec'
  cunit2 = 'arcsec'
  cunit3 = 'Angstrom'
; Axis labels:
  ctype1 = 'Solar-X'
  ctype2 = 'Solar-Y'
  ctype3 = 'Wavelength'
; rotation angles - should come from spacecraft data - TBC
  inst_rot = spcd-> getinst_rot()
; initial value, variations in binary table extension
  sat_rot = (spcd-> getsat_rot())[0]
; pointing, pick initial position of center of fov, 
; variations in binary table extension
  xcen = crval1 + cdelt1*((nraster+1)/2-crpix1)
  ycen = crval2 + cdelt2*((yw+1)/2-crpix2)
  ; if no spacecraft data available, sat_rot and inst_rot is set to 'NA'
  if is_string(sat_rot) then begin
    crota1 = 'NA'
    crota2 = 'NA'
  endif else begin
    crota1 = sat_rot + inst_rot
    crota2 = sat_rot + inst_rot
  endelse

  fxaddpar, header, 'xcen', xcen[0],  $
                    'Center pointing (Solar X)'
  fxaddpar, header, 'ycen', ycen[0], $
                    'Center pointing (Solar Y)'
  fxaddpar, header, 'cmirr', exp_info.cmirr_pos, $
                    'Coarse mirror position in unknown units'
; 20061114 viggoh: defined reference pixel to (1,512)
; 20100217 viggoh/johnm: redefined to (1,1)
  fxaddpar, header, 'crpix1', crpix1, $
                    'Reference pixel in solar-x'
  fxaddpar, header, 'crpix2', crpix2, $
                    'Reference pixel in solar-y'
  fxaddpar, header, 'crval1', crval1[0], $
                    'Coordinate value of CRPIX1'
  fxaddpar, header, 'crval2', crval2[0], $
                    'Coordinate value of CRPIX2'
; 20100217 viggoh/johnm: set to -1*stepsize since EIS rasters eastward
  fxaddpar, header, 'cdelt1',cdelt1, $
                    'Pixel size in solar-x'
  fxaddpar, header, 'cdelt2', cdelt2, $
                    'Pixel size in solar-y'
  fxaddpar, header, 'cdelt3', cdelt3, $
                    'Pixel size in wavelength (dispersion)'
  fxaddpar, header, 'cunit1', cunit1, $
                    'Definition of data unit for axis 1'
  fxaddpar, header, 'cunit2', cunit2, $
                    'Definition of data unit for axis 2'
  fxaddpar, header, 'cunit3', cunit3, $
                    'Definition of data unit for axis 3'
  fxaddpar, header, 'ctype1', ctype1, $
                    'Label for axis 1'
  fxaddpar, header, 'ctype2', ctype2, $
                    'Label for axis 2'
  fxaddpar, header, 'ctype3', ctype3, $
                    'Label for axis 3'
  fxaddpar, header, 'tr_mode', spcd-> gettr_mode(), $
                    'Spacecraft/instrument tracking mode'
  fxaddpar, header, 'inst_rot', inst_rot, $
                    'Instrument rotation angle'
  fxaddpar, header, 'sat_rot', sat_rot, $
                    'Satellite rotation angle'
  fxaddpar, header, 'crota1', crota1, $
                    'INST_ROT + SAT_ROT'
  fxaddpar, header, 'crota2', crota2, $
                    'INST_ROT + SAT_ROT'
  fxaddpar, header, 'fovx', fovx, $
                    'Field of view in Solar X (arcsec)'  
  fxaddpar, header, 'fovy', fovy, $
                    'Field of view in Solar Y (arcsec)'  
  fxaddpar, header, 'eistflxs',fix(exp_info.eis_x_sign), $
                    'Sign - EIS Transl. of XRT Flare X (0=Pos. 1=Neg)'
  fxaddpar, header, 'eistxflx', exp_info.eis_xrt_fl_x, $
                    'EIS Transl. of XRT Flare X-coord. (arcsec)'  
  fxaddpar, header, 'eistflys',fix(exp_info.eis_y_sign), $
                    'Sign - EIS Transl of XRT Flare Y (0=Pos. 1= Neg)'
  fxaddpar, header, 'eistxfly', exp_info.eis_xrt_fl_y, $
                    'EIS Transl. of XRT Flare Y-coord. (arcsec)'
  fxaddpar, header, 'eisfltx', exp_info.eis_fltr_x, $
                    'EIS flare trigger X-coordinate'
  fxaddpar, header, 'eisflty', exp_info.eis_fltr_y, $
                    'EIS flare trigger Y-coordinate'
  fxaddpar, header, 'eisevtx', exp_info.eis_evtr_x, $
                    'EIS event trigger X-location'
  fxaddpar, header, 'eisevty', exp_info.eis_evtr_y, $
                    'EIS event trigger Y-coordinate'
  fxaddpar, header, 'asrcdir', fix(exp_info.asrc_dir), $
                    'Anti Solar Rot. Comp. dir. (0=forw, 1=backw)'
  fxaddpar, header, 'asrcnr', fix(exp_info.asrc_nr), $
                    'ASRC number of FMIR steps'
  fxaddpar, header, 'asrcskip', fix(exp_info.asrc_skip), $
                    'ASRC frequency'
end
